/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.jee2.meta.proyecto.base;

import adalid.commons.properties.PropertiesHandler;
import adalid.commons.util.StrUtils;
import adalid.core.Project;
import adalid.core.enums.ProjectStage;
import adalid.core.enums.SecurityRealmType;
import adalid.core.interfaces.Entity;
import adalid.core.programmers.AbstractJavaProgrammer;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import meta.entidad.comun.configuracion.basica.Aplicacion;
import meta.entidad.comun.configuracion.basica.ClaseRecurso;
import meta.entidad.comun.configuracion.basica.Dominio;
import meta.entidad.comun.configuracion.basica.Funcion;
import meta.entidad.comun.configuracion.basica.FuncionParametro;
import meta.entidad.comun.configuracion.basica.GrupoProceso;
import meta.entidad.comun.configuracion.basica.Pagina;
import meta.entidad.comun.configuracion.basica.Parametro;
import meta.enumeracion.base.TipoModuloBase;
import meta.modulo.base.ModuloConsulta;
import meta.modulo.base.ModuloProcesamiento;
import meta.modulo.base.ModuloRegistro;
import meta.paquete.base.PaqueteBase;
import meta.paquete.comun.PaqueteConsultaRecursosBasicos;
import meta.paquete.comun.PaqueteRegistroRecursosBasicos;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class ProyectoJava1 extends ProyectoBase {

    private static final Logger logger = Logger.getLogger(Project.class);

    protected static final boolean TYPELESS_REALM_NAME = true;

    protected static final String VERSION_JAVA = "java.version";

    protected static final Map<String, String> ENBG = new LinkedHashMap<>(); // Entity Name Bundle Getters

    protected static final Set<String> PAKS; // Package Alias Keyword Set

    static {
        ENBG.put(Aplicacion.class.getSimpleName(), "BundleWebui.getNombreAplicacion");
        ENBG.put(ClaseRecurso.class.getSimpleName(), "BundleDominios.getNombreClaseRecurso");
        ENBG.put(Dominio.class.getSimpleName(), "BundleDominios.getNombreDominio");
//      ENBG.put(DominioParametro.class.getSimpleName(), "BundleParametros.getNombreDominioParametro");
        ENBG.put(Funcion.class.getSimpleName(), "BundleFunciones.getNombreFuncion");
        ENBG.put(FuncionParametro.class.getSimpleName(), "BundleParametros.getNombreFuncionParametro");
        ENBG.put(GrupoProceso.class.getSimpleName(), "BundleFunciones.getNombreGrupoProceso");
//      ENBG.put(OpcionMenu.class.getSimpleName(), "BundleMenus.getNombreOpcionMenu");
        ENBG.put(Pagina.class.getSimpleName(), "BundleWebui.getNombrePagina");
        ENBG.put(Parametro.class.getSimpleName(), "BundleParametros.getNombreParametro");
        /**/
        PAKS = AbstractJavaProgrammer.getJavaKeywords();
        PAKS.add("context");
        PAKS.add("pages");
    }

    public static String getEntityNameBundleGetter(Entity entity) {
        return entity == null ? null : getEntityNameBundleGetter(entity.getName());
    }

    public static String getEntityNameBundleGetter(String entityName) {
        return entityName == null ? null : ENBG.get(entityName);
    }

    public ProyectoJava1() {
        super();
    }

    protected ModuloConsulta consulta;

    protected ModuloProcesamiento procesamiento;

    protected ModuloRegistro registro;

    protected PaqueteConsultaRecursosBasicos consultaRecursosBasicos;

    protected PaqueteRegistroRecursosBasicos registroRecursosBasicos;

    @Override
    public boolean analyze() {
        boolean analyzed = super.analyze();
        analyzed &= analyzePackages();
        return analyzed;
    }

    private String _earProjectName;

    private String _ejbProjectName;

    private String _libProjectName;

    private String _resourcesProjectName;

    private String _webProjectName;

    private String _rootPackageName;

    private boolean _flushAfterEachInsert = true;

    private boolean _flushAfterEachUpdate = true;

    private boolean _flushAfterEachDelete = true;

    private boolean _internetAccessAllowed;

    private boolean _webServicesEnabled;

    private boolean _projectMailingEnabled;

    private boolean _projectRecaptchaEnabled;

    private boolean _exporterShellEnabled;

    private boolean _reporterShellEnabled;

    private boolean _sqlAgentShellEnabled;

    private ProjectStage _projectStage;

    private SecurityRealmType _securityRealmType;

    private String _securityRealmName;

    private String _roleBasedAccessControllerName;

    private boolean _authenticatedUserAutomaticRegistrationEnabled;

    private boolean analyzePackages() {
        boolean ok = true;
        String alias, dotted;
        String[] words;
        TipoModuloBase tipo;
        Set<String> names, allNames;
        Set<String> aliases = new LinkedHashSet<>();
        Set<String> consultas = new LinkedHashSet<>();
        Set<String> procesos = new LinkedHashSet<>();
        Set<String> registros = new LinkedHashSet<>();
        List<? extends PaqueteBase> packages = getPackages();
        for (PaqueteBase paquete : packages) {
            alias = paquete.getAlias();
            if (StringUtils.isBlank(alias)) {
                logger.error(paquete + " is invalid; its alias is null");
                ok = false;
            } else if (aliases.add(alias)) {
                dotted = StrUtils.getLowerCaseIdentifier(alias, '.');
                words = StringUtils.split(dotted, '.');
                for (String word : words) {
                    for (String keyword : PAKS) {
                        if (keyword.equalsIgnoreCase(word)) {
                            logger.error(paquete + " is invalid; its alias contains keyword \"" + keyword + "\"");
                            ok = false;
                        }
                    }
                }
            } else {
                logger.error(paquete + " is invalid; duplicate alias \"" + alias + "\"");
                ok = false;
            }
            names = paquete.getLocallyDeclaredEntityClassSimpleNames();
            if (names.isEmpty()) {
                logger.error(paquete + " is invalid; it has no entities locally declared");
                ok = false;
            }
            allNames = null;
            tipo = paquete.getTipo();
            if (tipo == null) {
                logger.error(paquete + " is invalid; its type is null");
                ok = false;
            } else {
                switch (tipo) {
                    case CONSULTA:
                        allNames = consultas;
                        break;
                    case PROCESAMIENTO:
                        allNames = procesos;
                        break;
                    case REGISTRO:
                        allNames = registros;
                        break;
                    default:
                        logger.error(paquete + " is invalid; its type is " + tipo);
                        ok = false;
                        break;
                }
            }
            if (allNames == null || names.isEmpty()) {
            } else {
                for (String name : names) {
                    if (allNames.add(name)) {
                    } else {
                        logger.error(name + " is declared in more than one package of type " + tipo);
                        ok = false;
                    }
                }
            }
        }
        return ok;
    }

    private List<? extends PaqueteBase> getPackages() {
        PaqueteBase paquete;
        List<PaqueteBase> packages = new ArrayList<>();
        List<Project> modules = getModulesList();
        for (Project module : modules) {
            if (module instanceof PaqueteBase) {
                paquete = (PaqueteBase) module;
                packages.add(paquete);
            }
        }
        return packages;
    }

    public void addBeanAttribute(String classSimpleName) {
        addBeanAttribute(classSimpleName, null);
    }

    public void addBeanAttribute(String classSimpleName, String beanName) {
        final String key = "bean.name";
        if (StringUtils.isNotBlank(classSimpleName)) {
            String cn = StringUtils.capitalize(classSimpleName.trim());
            String bn = StringUtils.uncapitalize(StringUtils.defaultIfBlank(StringUtils.trim(beanName), cn));
            addAttribute(cn + "." + key, bn);
        }
    }

    /**
     * @return the ear project name
     */
    public String getEarProjectName() {
        return StringUtils.defaultIfBlank(_earProjectName, getDefaultEarProjectName());
    }

    /**
     * @param earProjectName the ear project name to set
     */
    public void setEarProjectName(String earProjectName) {
        _earProjectName = StrUtils.getLowerCaseIdentifier(earProjectName, '-');
    }

    /**
     * @return the ejb project name
     */
    public String getEjbProjectName() {
        return StringUtils.defaultIfBlank(_ejbProjectName, getDefaultEjbProjectName());
    }

    /**
     * @param ejbProjectName the ejb project name to set
     */
    public void setEjbProjectName(String ejbProjectName) {
        _ejbProjectName = StrUtils.getLowerCaseIdentifier(ejbProjectName, '-');
    }

    /**
     * @return the lib project name
     */
    public String getLibProjectName() {
        return StringUtils.defaultIfBlank(_libProjectName, getDefaultLibProjectName());
    }

    /**
     * @param libProjectName the lib project name to set
     */
    public void setLibProjectName(String libProjectName) {
        _libProjectName = StrUtils.getLowerCaseIdentifier(libProjectName, '-');
    }

    /**
     * @return the resources project name
     */
    public String getResourcesProjectName() {
        return StringUtils.defaultIfBlank(_resourcesProjectName, getDefaultResourcesProjectName());
    }

    /**
     * @param resourcesProjectName the resources project name to set
     */
    public void setResourcesProjectName(String resourcesProjectName) {
        _resourcesProjectName = StrUtils.getLowerCaseIdentifier(resourcesProjectName, '-');
    }

    /**
     * @return the web project name
     */
    public String getWebProjectName() {
        return StringUtils.defaultIfBlank(_webProjectName, getDefaultWebProjectName());
    }

    /**
     * @param webProjectName the web project name to set
     */
    public void setWebProjectName(String webProjectName) {
        _webProjectName = StrUtils.getLowerCaseIdentifier(webProjectName, '-');
    }

    /**
     * @return the root package name
     */
    public String getRootPackageName() {
        return StringUtils.defaultIfBlank(_rootPackageName, getDefaultRootPackageName());
    }

    /**
     * @param rootPackageName the root package name to set
     */
    public void setRootPackageName(String rootPackageName) {
        _rootPackageName = StrUtils.getLowerCaseIdentifier(rootPackageName, '.');
    }

    /**
     * @return true if the database is defined with a national character set; false otherwise
     */
    public boolean isDatabaseNationalCharacterSet() {
        return PropertiesHandler.isDatabaseNationalCharacterSet();
    }

    /**
     * @return true if flush must be executed after each insert; false otherwise
     */
    public boolean isFlushAfterEachInsert() {
        return _flushAfterEachInsert;
    }

    /**
     * El método setFlushAfterEachInsert se utiliza para especificar si el proyecto generado debe sincronizar, o no, el contexto de persistencia
     * después de cada persist correspondiente al insert de una fila en una página de CRUD. El valor predeterminado de esta propiedad es true (se
     * sincroniza después de cada persist).
     *
     * @param enabled true, si el proyecto generado debe sincronizar el contexto de persistencia después de cada persist; de lo contrario false.
     */
    public void setFlushAfterEachInsert(boolean enabled) {
        _flushAfterEachInsert = enabled;
    }

    /**
     * @return true if flush must be executed after each update; false otherwise
     */
    public boolean isFlushAfterEachUpdate() {
        return _flushAfterEachUpdate;
    }

    /**
     * El método setFlushAfterEachUpdate se utiliza para especificar si el proyecto generado debe sincronizar, o no, el contexto de persistencia
     * después de cada merge correspondiente al update en una fila en una página de CRUD. El valor predeterminado de esta propiedad es true (se
     * sincroniza después de cada merge).
     *
     * @param enabled true, si el proyecto generado debe sincronizar el contexto de persistencia después de cada merge; de lo contrario false.
     */
    public void setFlushAfterEachUpdate(boolean enabled) {
        _flushAfterEachUpdate = enabled;
    }

    /**
     * @return true if flush must be executed after each delete; false otherwise
     */
    public boolean isFlushAfterEachDelete() {
        return _flushAfterEachDelete;
    }

    /**
     * El método setFlushAfterEachDelete se utiliza para especificar si el proyecto generado debe sincronizar, o no, el contexto de persistencia
     * después de cada remove correspondiente al delete de una fila en una página de CRUD. El valor predeterminado de esta propiedad es true (se
     * sincroniza después de cada remove).
     *
     * @param enabled true, si el proyecto generado debe sincronizar el contexto de persistencia después de cada remove; de lo contrario false.
     */
    public void setFlushAfterEachDelete(boolean enabled) {
        _flushAfterEachDelete = enabled;
    }

    /**
     * @return true if internet access should be allowed; false otherwise
     */
    public boolean isInternetAccessAllowed() {
        return _internetAccessAllowed;
    }

    /**
     * El método setInternetAccessAllowed se utiliza para especificar si el proyecto generado incluye, o no, características que requieren acceso a
     * internet para su funcionamiento (por ejemplo, si la página Inicio de Sesión incluye, o no, un elemento Google reCaptcha). El valor
     * predeterminado de esta propiedad es false (no se incluyen características que requieren acceso a internet).
     *
     * @param allowed true, si el proyecto generado incluye características que requieren acceso a internet; de lo contrario false.
     */
    public void setInternetAccessAllowed(boolean allowed) {
        _internetAccessAllowed = allowed;
    }

    /**
     * @return true if web services should be disabled; false otherwise
     */
    public boolean isWebServicesDisabled() {
        return !isWebServicesEnabled();
    }

    /**
     * @return true if web services should be enabled; false otherwise
     */
    public boolean isWebServicesEnabled() {
        return _webServicesEnabled;
    }

    /**
     * El método setWebServicesEnabled se utiliza para especificar si el proyecto generado incluye, o no, servicios web. El valor predeterminado de
     * esta propiedad es false (no se incluyen servicios web).
     *
     * @param enabled true, si el proyecto generado incluye servicios web; de lo contrario false.
     */
    public void setWebServicesEnabled(boolean enabled) {
        _webServicesEnabled = enabled;
    }

    /**
     * @return true if project mailing should be enabled; false otherwise
     */
    public boolean isProjectMailingEnabled() {
        return _projectMailingEnabled;
    }

    /**
     * El método setProjectMailingEnabled se utiliza para especificar si el proyecto generado incluye, o no, características que requieren acceso a un
     * servidor de correo electrónico (e-mail) para su funcionamiento (por ejemplo, si las notificaciones de tareas se envían, o no, por correo
     * electrónico). El valor predeterminado de esta propiedad es false (no se incluyen características que requieren acceso a un servidor de correo
     * electrónico).
     *
     * @param enabled true, si el proyecto generado incluye características que requieren acceso a un servidor de correo electrónico; de lo contrario
     * false.
     */
    public void setProjectMailingEnabled(boolean enabled) {
        _projectMailingEnabled = enabled;
    }

    /**
     * @return true if project recaptcha should be enabled; false otherwise
     */
    public boolean isProjectRecaptchaEnabled() {
        return _projectRecaptchaEnabled;
    }

    /**
     * El método setProjectRecaptchaEnabled se utiliza para especificar si el proyecto generado utiliza, o no, Google reCAPTCHA. El valor
     * predeterminado de esta propiedad es false (no se utiliza Google reCAPTCHA).
     *
     * @param enabled true, si el proyecto generado utiliza Google reCAPTCHA; de lo contrario false.
     */
    public void setProjectRecaptchaEnabled(boolean enabled) {
        _projectRecaptchaEnabled = enabled;
    }

    /**
     * @return true if exporter shell should be enabled; false otherwise
     */
    public boolean isExporterShellEnabled() {
        return _exporterShellEnabled;
    }

    /**
     * El método setExporterShellEnabled se utiliza para especificar si el proyecto generado utiliza, o no, procesos nativos del sistema operativo
     * para exportar archivos. El valor predeterminado de esta propiedad es false (no se utilizan procesos nativos).
     *
     * @param enabled true, si el proyecto generado utiliza procesos nativos del sistema operativo para exportar archivos; de lo contrario false.
     */
    public void setExporterShellEnabled(boolean enabled) {
        _exporterShellEnabled = enabled;
    }

    /**
     * @return true if reporter shell should be enabled; false otherwise
     */
    public boolean isReporterShellEnabled() {
        return _reporterShellEnabled;
    }

    /**
     * El método setReporterShellEnabled se utiliza para especificar si el proyecto generado utiliza, o no, procesos nativos del sistema operativo
     * para producir informes. El valor predeterminado de esta propiedad es false (no se utilizan procesos nativos).
     *
     * @param enabled true, si el proyecto generado utiliza procesos nativos del sistema operativo para producir informes; de lo contrario false.
     */
    public void setReporterShellEnabled(boolean enabled) {
        _reporterShellEnabled = enabled;
    }

    /**
     * @return true if sql agent shell should be enabled; false otherwise
     */
    public boolean isSqlAgentShellEnabled() {
        return _sqlAgentShellEnabled;
    }

    /**
     * El método setSqlAgentShellEnabled se utiliza para especificar si el proyecto generado utiliza, o no, procesos nativos del sistema operativo
     * para ejecutar procedimientos almacenados en la base de datos. El valor predeterminado de esta propiedad es false (no se utilizan procesos
     * nativos).
     *
     * @param enabled true, si el proyecto generado utiliza procesos nativos del sistema operativo para ejecutar procedimientos almacenados en la base
     * de datos; de lo contrario false.
     */
    public void setSqlAgentShellEnabled(boolean enabled) {
        _sqlAgentShellEnabled = enabled;
    }

    /**
     * @return the project stage
     */
    public ProjectStage getProjectStage() {
        return _projectStage == null ? getDefaultProjectStage() : _projectStage;
    }

    /**
     * El método setProjectStage se utiliza para especificar la etapa en la que se encuentra el proyecto generado. El valor puede ser cualquiera de
     * los elementos de la enumeración ProjectStage, es decir: DEVELOPMENT, TESTING, ACCEPTANCE, o PRODUCTION. El valor predeterminado de esta
     * propiedad es DEVELOPMENT.
     *
     * @param stage etapa en la que se encuentra el proyecto generado; especifique DEVELOPMENT, TESTING, ACCEPTANCE, o PRODUCTION si el proyecto está
     * en Desarrollo, Pruebas, Pruebas de Aceptación o Producción, respectivamente.
     */
    public void setProjectStage(ProjectStage stage) {
        _projectStage = stage;
    }

    /**
     * @return the security realm type
     */
    public SecurityRealmType getSecurityRealmType() {
        return _securityRealmType == null ? getDefaultSecurityRealmType() : _securityRealmType;
    }

    /**
     * El método setSecurityRealmType se utiliza para establecer el tipo de dominio de seguridad del proyecto. El valor predeterminado de esta
     * propiedad es JDBC.
     *
     * @param securityRealmType elemento de la enumeración SecurityRealmType que corresponde el tipo de dominio de seguridad del proyecto. Especifique
     * LDAP si el controlador de seguridad del proyecto cumple el protocolo LDAP. En caso contrario especifique JDBC.
     */
    public void setSecurityRealmType(SecurityRealmType securityRealmType) {
        _securityRealmType = securityRealmType;
    }

    /**
     * @return the security realm name
     */
    public String getSecurityRealmName() {
        return StringUtils.defaultIfBlank(_securityRealmName, getDefaultSecurityRealmName());
    }

    /**
     * @param securityRealmName the security realm name to set
     */
    public void setSecurityRealmName(String securityRealmName) {
        _securityRealmName = StrUtils.getLowerCaseIdentifier(securityRealmName, '-');
    }

    /**
     * @return the role-based-access-controller (RBAC) name
     */
    public String getRoleBasedAccessControllerName() {
        return StringUtils.defaultIfBlank(_roleBasedAccessControllerName, getDefaultRoleBasedAccessControllerName());
    }

    /**
     * El método setRoleBasedAccessControllerName se utiliza para establecer el nombre del controlador de seguridad del proyecto. El valor
     * predeterminado de esta propiedad es el alias del proyecto.
     *
     * @param roleBasedAccessControllerName nombre del controlador de seguridad del proyecto. Especifique LDAP si el controlador de seguridad del
     * proyecto cumple el protocolo LDAP y desea que el controlador LDAP, además de la autenticación, también controle la asignación de los roles de
     * los usuarios; si desea que el controlador LDAP controle solo la autenticación y la aplicación controle la asignación de los roles de los
     * usuarios, entonces especifique el alias del proyecto.
     */
    public void setRoleBasedAccessControllerName(String roleBasedAccessControllerName) {
        _roleBasedAccessControllerName = StrUtils.getIdentifier(roleBasedAccessControllerName);
    }

    /**
     * @return true if authenticated users automatic registration should be enabled; false otherwise
     */
    public boolean isAuthenticatedUserAutomaticRegistrationEnabled() {
        return _authenticatedUserAutomaticRegistrationEnabled && !SecurityRealmType.JDBC.equals(getSecurityRealmType());
    }

    /**
     * El método setAuthenticatedUserAutomaticRegistrationEnabled se utiliza para especificar si el proyecto generado debe registrar automáticamente,
     * o no, los usuarios que se conectan y no están registrados en la base de datos. Un usuario se puede conectar sin estar registrado en la base de
     * datos solo si el tipo de dominio de seguridad del proyecto no es JDBC.
     *
     * @param enabled true, si el proyecto generado debe registrar automáticamente, o no, los usuarios que se conectan y no están registrados en la
     * base de datos; de lo contrario false.
     */
    public void setAuthenticatedUserAutomaticRegistrationEnabled(boolean enabled) {
        _authenticatedUserAutomaticRegistrationEnabled = enabled;
    }

    protected String getDefaultEarProjectName() {
        return getAlias();
    }

    protected String getDefaultEjbProjectName() {
        return getAlias() + "-ejb";
    }

    protected String getDefaultLibProjectName() {
        return getAlias() + "-lib";
    }

    protected String getDefaultResourcesProjectName() {
        return getAlias() + "-resources";
    }

    protected String getDefaultWebProjectName() {
        return getAlias() + "-war";
    }

    protected String getDefaultRootPackageName() {
        return getAlias();
    }

    protected ProjectStage getDefaultProjectStage() {
        return ProjectStage.DEVELOPMENT;
    }

    protected SecurityRealmType getDefaultSecurityRealmType() {
        return SecurityRealmType.JDBC;
    }

    protected String getDefaultSecurityRealmName() {
        return getAlias() + "-" + (TYPELESS_REALM_NAME ? "auth" : getSecurityRealmType().name().toLowerCase()) + "-" + "realm";
    }

    protected String getDefaultRoleBasedAccessControllerName() {
        /*
        El valor predeterminado de esta propiedad es el nombre del tipo de dominio de seguridad del proyecto (vea Método setSecurityRealmType).
        return getSecurityRealmType().name();
        **/
        return getAlias();
    }

}
