/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.jee2.meta.proyecto.base;

import adalid.commons.util.BitUtils;
import adalid.commons.util.StrUtils;
import meta.psm.PrimeFacesThemes;
import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public abstract class ProyectoMavenPrime extends ProyectoMaven {

    private static final Logger logger = Logger.getLogger(ProyectoMavenPrime.class);

    protected static final String PRIMEFACES_THEME = "primefaces.theme";

    protected static final String PRIMEFACES_THEME_VERSION = "primefaces.theme.version";

    protected static final String PRIMEFACES_THEME_SWITCHING_ENABLED = "primefaces.theme.switching.enabled";

    protected static final String DEFAULT_THEME = PrimeFacesThemes.HOT_SNEAKS;

    protected static final String DEFAULT_THEME_VERSION = "[1.0.0,)";

    private String _theme;

    private String _themeVersion;

    private Boolean _themeSwitchingEnabled;

    @Override
    protected void loadPrivateProperties(Level level, ExtendedProperties properties) {
        super.loadPrivateProperties(level, properties);
        if (_theme == null) {
            _theme = properties.getString(PRIMEFACES_THEME);
            logger.log(level, PRIMEFACES_THEME + "=" + _theme);
        }
        if (_themeVersion == null) {
            _themeVersion = properties.getString(PRIMEFACES_THEME_VERSION);
            logger.log(level, PRIMEFACES_THEME_VERSION + "=" + _themeVersion);
        }
        if (_themeSwitchingEnabled == null) {
            _themeSwitchingEnabled = BitUtils.valueOf(properties.getString(PRIMEFACES_THEME_SWITCHING_ENABLED));
            logger.log(level, PRIMEFACES_THEME_SWITCHING_ENABLED + "=" + _themeSwitchingEnabled);
        }
    }

    /**
     * @return the theme
     */
    public String getTheme() {
        return StringUtils.defaultIfBlank(_theme, getDefaultTheme());
    }

    /**
     * El método setTheme se utiliza para especificar si el nombre del artefacto Maven del tema de interfaz gráfica del proyecto generado. El valor
     * predeterminado de esta propiedad es PrimeFacesThemes.HOT_SNEAKS (campo HOT_SNEAKS de la clase PrimeFacesThemes).
     *
     * @param theme nombre del artefacto Maven del tema de interfaz gráfica del proyecto generado; si el proyecto generado permite cambiar el tema,
     * este será el nombre del tema inicial. Utilice como argumento alguno de los campos definidos en la clase PrimeFacesThemes.
     */
    public void setTheme(String theme) {
        _theme = StrUtils.getMavenIdentifier(theme);
    }

    protected String getDefaultTheme() {
        return DEFAULT_THEME;
    }

    /**
     * @return the theme version
     */
    public String getThemeVersion() {
        return StringUtils.defaultIfBlank(_themeVersion, getDefaultThemeVersion());
    }

    /**
     * El método setThemeVersion se utiliza para especificar la versión del artefacto Maven del tema de interfaz gráfica del proyecto generado. El
     * valor predeterminado de esta propiedad es [1.0.0,) (versión 1.0.0 en adelante).
     *
     * @param version versión del artefacto Maven del tema de la interfaz gráfica del proyecto generado; si el proyecto generado permite cambiar el
     * tema, esta será la versión del artefacto all-themes.
     */
    public void setThemeVersion(String version) {
        _themeVersion = StrUtils.getMavenVersion(version);
    }

    protected String getDefaultThemeVersion() {
        return DEFAULT_THEME_VERSION;
    }

    /**
     * @return true if the theme switching is enabled; false otherwise
     */
    public boolean isThemeSwitchingEnabled() {
        return BitUtils.valueOf(_themeSwitchingEnabled);
    }

    /**
     * El método setThemeSwitchingEnabled se utiliza para especificar si el proyecto generado permite, o no, cambiar el tema de la interfaz gráfica.
     * El valor predeterminado de esta propiedad es false (no permite cambiar el tema).
     *
     * @param enabled true, si el proyecto generado permite cambiar el tema de la interfaz gráfica; de lo contrario false.
     */
    public void setThemeSwitchingEnabled(boolean enabled) {
        _themeSwitchingEnabled = enabled;
    }

}
