/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package meta.entidad.comun.operacion.basica;

import adalid.core.AbstractPersistentEntity;
import adalid.core.Key;
import adalid.core.annotations.Allocation;
import adalid.core.annotations.ColumnField;
import adalid.core.annotations.EntityClass;
import adalid.core.annotations.EntityCodeGen;
import adalid.core.annotations.EntityConsoleView;
import adalid.core.annotations.EntityDeleteOperation;
import adalid.core.annotations.EntityDetailView;
import adalid.core.annotations.EntityInsertOperation;
import adalid.core.annotations.EntityReferenceSearch;
import adalid.core.annotations.EntitySelectOperation;
import adalid.core.annotations.EntityTableView;
import adalid.core.annotations.EntityTreeView;
import adalid.core.annotations.EntityTriggers;
import adalid.core.annotations.EntityUpdateOperation;
import adalid.core.annotations.EntityWarnings;
import adalid.core.annotations.ForeignKey;
import adalid.core.annotations.ManyToOne;
import adalid.core.annotations.NameProperty;
import adalid.core.annotations.PrimaryKey;
import adalid.core.annotations.PropertyAggregation;
import adalid.core.annotations.PropertyField;
import adalid.core.annotations.StringField;
import adalid.core.annotations.VersionProperty;
import adalid.core.enums.AggregateFunction;
import adalid.core.enums.DefaultCondition;
import adalid.core.enums.Kleenean;
import adalid.core.enums.ListStyle;
import adalid.core.enums.MasterDetailView;
import adalid.core.enums.Navigability;
import adalid.core.enums.OnDeleteAction;
import adalid.core.enums.OnUpdateAction;
import adalid.core.enums.QuickAddingFilter;
import adalid.core.enums.ResourceGender;
import adalid.core.enums.ResourceType;
import adalid.core.enums.SearchType;
import adalid.core.interfaces.Artifact;
import adalid.core.interfaces.CharacterExpression;
import adalid.core.interfaces.Check;
import adalid.core.properties.BooleanProperty;
import adalid.core.properties.IntegerProperty;
import adalid.core.properties.LongProperty;
import adalid.core.properties.StringProperty;
import java.lang.reflect.Field;
import meta.entidad.comun.configuracion.basica.FuncionParametro;

/**
 * @author Jorge Campins
 */
@EntityClass(base = Kleenean.TRUE, independent = Kleenean.FALSE, resourceType = ResourceType.OPERATION, resourceGender = ResourceGender.FEMININE)
@EntityCodeGen(bws = Kleenean.FALSE, fws = Kleenean.FALSE)
@EntitySelectOperation(enabled = Kleenean.TRUE)
@EntityInsertOperation(enabled = Kleenean.TRUE)
@EntityUpdateOperation(enabled = Kleenean.TRUE)
@EntityDeleteOperation(enabled = Kleenean.TRUE)
@EntityTableView(enabled = Kleenean.FALSE)
@EntityDetailView(enabled = Kleenean.FALSE)
@EntityTreeView(enabled = Kleenean.FALSE)
@EntityConsoleView(enabled = Kleenean.FALSE)
@EntityTriggers(afterValue = Kleenean.TRUE)
@EntityWarnings(enabled = Kleenean.FALSE)
public class VistaFuncionCol extends AbstractPersistentEntity {

    // <editor-fold defaultstate="collapsed" desc="class constructors">
    @Deprecated
    private VistaFuncionCol() {
        this(null, null);
    }

    public VistaFuncionCol(Artifact declaringArtifact, Field declaringField) {
        super(declaringArtifact, declaringField);
    }
    // </editor-fold>

    @PrimaryKey
    public LongProperty id;

    @VersionProperty
    public LongProperty version;

    @NameProperty
    @PropertyField(hidden = Kleenean.TRUE, defaultCondition = DefaultCondition.UNCONDITIONALLY)
    @StringField(maxLength = 200)
    public StringProperty nombre;

    @ForeignKey(onDelete = OnDeleteAction.CASCADE, onUpdate = OnUpdateAction.CASCADE)
    @ManyToOne(navigability = Navigability.BIDIRECTIONAL, view = MasterDetailView.TABLE_AND_DETAIL)
    @ColumnField(nullable = Kleenean.FALSE)
    @Allocation(maxDepth = 2, maxRound = 0)
    public VistaFuncion vista;

//  20171213: remove foreign-key referring to FuncionParametro
//  @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(navigability = Navigability.UNIDIRECTIONAL, view = MasterDetailView.NONE, quickAdding = QuickAddingFilter.MISSING)
    @ColumnField(nullable = Kleenean.FALSE)
    @EntityReferenceSearch(searchType = SearchType.LIST, listStyle = ListStyle.NAME)
    @PropertyField(required = Kleenean.TRUE, table = Kleenean.TRUE, create = Kleenean.TRUE, update = Kleenean.FALSE)
    @Allocation(maxDepth = 2, maxRound = 0)
    public FuncionParametro columna;

    /**
     * string property field
     */
    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE, table = Kleenean.FALSE, defaultCondition = DefaultCondition.UNCONDITIONALLY)
    @StringField(maxLength = 200)
    public StringProperty alias;

    /**
     * string property field
     */
    @PropertyField(table = Kleenean.TRUE, create = Kleenean.TRUE)
    @StringField(maxLength = 30)
    public StringProperty etiqueta;

    /**
     * integer property field
     */
    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(required = Kleenean.FALSE, table = Kleenean.TRUE, create = Kleenean.TRUE)
    public IntegerProperty secuencia;

    /**
     * many-to-one entity reference property field
     */
    @Allocation(maxDepth = 1, maxRound = 0)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(navigability = Navigability.UNIDIRECTIONAL, view = MasterDetailView.NONE)
    @PropertyField(required = Kleenean.FALSE, table = Kleenean.TRUE, create = Kleenean.TRUE)
    public TipoAgregacion agregacion;

    /**
     * many-to-one entity reference property field
     */
//  @Allocation(maxDepth = 3, maxRound = 2)
    @Allocation(maxDepth = 3, maxRound = 3)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(navigability = Navigability.UNIDIRECTIONAL, view = MasterDetailView.NONE)
    @EntityReferenceSearch(searchType = SearchType.LIST, listStyle = ListStyle.NAME)
    @PropertyField(required = Kleenean.FALSE, table = Kleenean.TRUE, create = Kleenean.TRUE)
    public VistaFuncionCol grupo;

    /**
     * boolean property field
     */
    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(required = Kleenean.FALSE, table = Kleenean.TRUE, create = Kleenean.TRUE)
    public BooleanProperty orden;

    /**
     * boolean property field
     */
    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(hidden = Kleenean.TRUE)
    public BooleanProperty visible;

    /**
     * boolean property field
     */
    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(hidden = Kleenean.TRUE)
    public BooleanProperty graficable;

    /**
     * integer property field
     */
    @ColumnField(nullable = Kleenean.TRUE)
    @PropertyField(hidden = Kleenean.TRUE)
    public IntegerProperty pixeles;

    /**
     * integer property field
     */
    @ColumnField(nullable = Kleenean.TRUE)
    @PropertyField(required = Kleenean.FALSE, table = Kleenean.TRUE, create = Kleenean.TRUE)
    @PropertyAggregation(function = AggregateFunction.CUSTOM_MADE, title = "ancho disponible para columnas del detalle")
    public IntegerProperty anchoPorMil;

    @Override
    protected void settleAttributes() {
        super.settleAttributes();
//      setSchema(ProyectoBase.getEsquemaEntidadesComunes());
        setOrderBy(vista, secuencia, id);
        // <editor-fold defaultstate="collapsed" desc="localization of VistaFuncionCol's attributes">
        setLocalizedLabel(ENGLISH, "view column");
        setLocalizedLabel(SPANISH, "columna de vista");
        setLocalizedShortLabel(ENGLISH, "column");
        setLocalizedShortLabel(SPANISH, "columna");
        setLocalizedCollectionLabel(ENGLISH, "View Columns");
        setLocalizedCollectionLabel(SPANISH, "Columnas de Vista");
        setLocalizedCollectionShortLabel(ENGLISH, "Columns");
        setLocalizedCollectionShortLabel(SPANISH, "Columnas");
        setLocalizedDescription(ENGLISH, "column of view defined by the end user");
        setLocalizedDescription(SPANISH, "columna de vista definida por el usuario final");
        // </editor-fold>
    }

    @Override
    protected void settleProperties() {
        super.settleProperties();
        nombre.setDefaultValue(columna.nombreFuncionParametro);
        alias.setDefaultValue(columna.codigoFuncionParametro);
        secuencia.setMinValue(0);
        secuencia.setMaxValue(10000);
        orden.setInitialValue(false);
        orden.setDefaultValue(false);
        visible.setInitialValue(true);
        visible.setDefaultValue(true);
        graficable.setInitialValue(true);
        graficable.setDefaultValue(true);
        pixeles.setDefaultValue(columna.parametro.pixeles);
        pixeles.setMinValue(0);
        pixeles.setMaxValue(960);
        anchoPorMil.setDefaultValue(columna.parametro.pixeles.times(1000).over(960).toInteger());
        anchoPorMil.setMinValue(0);
        anchoPorMil.setMaxValue(1000);
        // <editor-fold defaultstate="collapsed" desc="localization of VistaFuncionCol's properties">
        nombre.setLocalizedLabel(ENGLISH, "view column name");
        nombre.setLocalizedLabel(SPANISH, "nombre de la columna de vista");
        nombre.setLocalizedShortLabel(ENGLISH, "name");
        nombre.setLocalizedShortLabel(SPANISH, "nombre");
        /**/
        vista.setLocalizedLabel(ENGLISH, "view");
        vista.setLocalizedLabel(SPANISH, "vista");
        /**/
        columna.setLocalizedLabel(ENGLISH, "column");
        columna.setLocalizedLabel(SPANISH, "columna");
        /**/
        alias.setLocalizedLabel(ENGLISH, "alias");
        alias.setLocalizedLabel(SPANISH, "alias");
        /**/
        etiqueta.setLocalizedLabel(ENGLISH, "label");
        etiqueta.setLocalizedLabel(SPANISH, "etiqueta");
        /**/
        secuencia.setLocalizedLabel(ENGLISH, "sequence");
        secuencia.setLocalizedLabel(SPANISH, "secuencia");
        /**/
        agregacion.setLocalizedLabel(ENGLISH, "aggregation");
        agregacion.setLocalizedLabel(SPANISH, "agregación");
        /**/
        grupo.setLocalizedLabel(ENGLISH, "group");
        grupo.setLocalizedLabel(SPANISH, "grupo");
        /**/
        orden.setLocalizedLabel(ENGLISH, "order");
        orden.setLocalizedLabel(SPANISH, "orden");
        /**/
        visible.setLocalizedLabel(ENGLISH, "visible");
        visible.setLocalizedLabel(SPANISH, "visible");
        /**/
        graficable.setLocalizedLabel(ENGLISH, "chartable");
        graficable.setLocalizedLabel(SPANISH, "graficable");
        /**/
        pixeles.setLocalizedLabel(ENGLISH, "width");
        pixeles.setLocalizedLabel(SPANISH, "ancho");
        pixeles.setLocalizedTooltip(ENGLISH, "width of the column in pixels");
        pixeles.setLocalizedTooltip(SPANISH, "anchura de la columna expresada en pixeles");
        /**/
        anchoPorMil.setLocalizedLabel(ENGLISH, "width");
        anchoPorMil.setLocalizedLabel(SPANISH, "ancho");
        anchoPorMil.setLocalizedTooltip(ENGLISH, "column width as a fraction of 1000");
        anchoPorMil.setLocalizedTooltip(SPANISH, "anchura de la columna expresada como una fracción de 1000");
        // </editor-fold>
    }

    protected Key uk_vista_funcion_col_0001;

    @Override
    protected void settleKeys() {
        super.settleKeys();
        uk_vista_funcion_col_0001.setUnique(true);
        uk_vista_funcion_col_0001.newKeyField(vista, columna);
    }

    @Override
    protected void settleLinks() {
        super.settleLinks();
        linkForeignOwnerProperty(vista.propietario);
    }

    protected Check check101;

    @Override
    protected void settleExpressions() {
        super.settleExpressions();
        /*
        check011 = columna.esParametroVinculado.implies(agregacion.isNullOrEqualTo(agregacion.GRUPO));
        /**/
        check101 = grupo.isNotNull().implies(grupo.isNotEqualTo(this));
        // <editor-fold defaultstate="collapsed" desc="localization of VistaFuncionCol's expressions">
        /*
        check011.setLocalizedLabel(ENGLISH, "check aggregation");
        check011.setLocalizedLabel(SPANISH, "chequear agregacion");
        check011.setLocalizedDescription(ENGLISH, "only Group can be specified if the column is linked");
        check011.setLocalizedDescription(SPANISH, "solo se puede especificar Grupo si la columna es vinculada");
        check011.setLocalizedErrorMessage(ENGLISH, "invalid aggregation: only Group can be specified if the column is linked");
        check011.setLocalizedErrorMessage(SPANISH, "agregación inválida: solo se puede especificar Grupo si la columna es vinculada");
        /**/
        check101.setLocalizedLabel(ENGLISH, "check group");
        check101.setLocalizedLabel(SPANISH, "chequear grupo");
        check101.setLocalizedDescription(ENGLISH, "the group can not be the same column");
        check101.setLocalizedDescription(SPANISH, "el grupo no puede ser la misma columna");
        check101.setLocalizedErrorMessage(ENGLISH, "the group is the same column");
        check101.setLocalizedErrorMessage(SPANISH, "el grupo es la misma columna");
        /**/
        // </editor-fold>
    }

    @Override
    protected void settleFilters() {
        super.settleFilters();
        columna.setSearchQueryFilter(columna.funcion.isEqualTo(vista.funcion).and(columna.parametro.pixeles.isGreaterThan(0)));
        CharacterExpression rango = columna.rangoAgregacion.numero.toCharString();
        agregacion.setSearchQueryFilter(agregacion.rangos.contains(rango));
        grupo.setSearchQueryFilter(grupo.agregacion.isEqualTo(agregacion.GRUPO).and(grupo.vista.isEqualTo(vista)));
    }

}
