/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.commons.properties;

import adalid.commons.util.*;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public class Dictionary {

    private static final Logger logger = Logger.getLogger(Dictionary.class);

    private static final ExtendedProperties bootstrapping = PropertiesHandler.getBootstrapping();

    private static final String DIR = System.getProperty("user.dir");

    private static final String SEP = System.getProperty("file.separator");

    private static final String LINE_SEPARATOR = System.lineSeparator();

    private static final String DFLS = bootstrapping.getString("sql.dictionary.file.line.separator", "UNSPECIFIED");

    private static final String DICTIONARY_FILE_LINE_SEPARATOR
        = StringUtils.isBlank(DFLS) ? LINE_SEPARATOR : DFLS.equalsIgnoreCase("CRLF") ? "\r\n" : DFLS.equalsIgnoreCase("LF") ? "\n" : LINE_SEPARATOR;

    private static final String RESOURCES = DIR + SEP + bootstrapping.getString("sql.dictionary.path", "resources").replace("/", SEP);

    private static final String DICTIONARY_DIR = RESOURCES + SEP + "dictionary";

    private static final String PROPERTIES_SUFFIX = ".properties";

    private static int errorCount, warningCount;

    private static Level _infoLevel = Level.OFF;

    private static boolean _fairInfoLevel;

    public static Level getInfoLevel() {
        return _infoLevel;
    }

    public static void setInfoLevel(Level level) {
        _infoLevel = LogUtils.check(level, Level.OFF, Level.INFO);
        _fairInfoLevel = LogUtils.fair(_infoLevel);
    }

    public static Dictionary load(Class<?> clazz, String folder) {
        return new Dictionary(clazz, folder);
    }

    public static boolean isValidNumericCode(String value) {
        return LongUtils.valueOf(value, 0L) != 0L;
    }

    public static void printSummary() {
        if (warningCount > 0) {
            logger.warn("warnings=" + warningCount);
        }
        if (errorCount > 0) {
            logger.warn("errors=" + errorCount);
        }
    }

    public static int getErrorCount() {
        return errorCount;
    }

    public static int getWarningCount() {
        return warningCount;
    }

    public static void reset() {
        errorCount = 0;
        warningCount = 0;
    }

    private final String simpleName, fileName, filePath;

    private final Map<String, String> map = new TreeMap<>();

    private final Properties properties;

    private final Set<String> errors = new TreeSet<>();

    private final Set<String> warnings = new TreeSet<>();

    private Dictionary(Class<?> clazz, String folder) {
        simpleName = clazz.getSimpleName();
        fileName = simpleName + PROPERTIES_SUFFIX;
        filePath = DICTIONARY_DIR + (folder != null && folder.matches("^\\w+$") ? SEP + folder + SEP : SEP) + fileName;
        properties = PropertiesHandler.loadProperties(filePath, true, Level.DEBUG);
        log(true);
        check();
    }

    private void check() {
        String value;
        for (Object key : properties.keySet()) {
            value = properties.getProperty("" + key);
            if (!isValidNumericCode(value)) {
                if (StringUtils.isBlank(value)) {
                    warn("property " + simpleName + " [" + key + "] value is missing");
                } else {
                    warn("\"" + value + "\" is an invalid numeric code; property " + simpleName + " [" + key + "] value will be replaced");
                }
            }
        }
    }

    public String getProperty(String key) {
        return properties.getProperty(key);
    }

    public Object setProperty(String key, String value) {
        Object previousValue = properties.setProperty(key, value);
        if (previousValue != null && !previousValue.equals(value)) {
            if (StringUtils.isBlank("" + previousValue)) {
                warn("assigning value to property " + simpleName + " [" + key + "]");
            } else {
                warn("replacing value of property " + simpleName + " [" + key + "]");
            }
        }
        putProperty(key, value);
        return previousValue;
    }

    public String putProperty(String key, String value) {
        String previousKey = map.put(value, key);
        if (previousKey != null && !previousKey.equals(key)) {
            int c = key.compareTo(previousKey);
            String k1 = c < 0 ? key : previousKey;
            String k2 = c < 0 ? previousKey : key;
            error("\"" + value + "\" is the same numeric code for properties " + simpleName + " [" + k1 + "] and [" + k2 + "]");
        }
        return previousKey;
    }

    public void store() {
        PropertiesHandler.storeProperties(properties, filePath, null, DICTIONARY_FILE_LINE_SEPARATOR);
        log(false);
    }

    private void log(boolean initializing) {
        String path = StringUtils.substringAfter(filePath, DIR).replace(SEP, "/");
        if (properties.isEmpty()) {
            if (initializing) {
                info("initializing " + path);
            } else {
                warn("file " + path + " is empty");
            }
        } else {
            info("file " + path + " has " + properties.size() + " properties");
        }
    }

    private void info(String message) {
        if (_fairInfoLevel) {
            logger.log(_infoLevel, message);
        }
    }

    private void warn(String message) {
        if (warnings.add(message)) {
            warningCount++;
            logger.warn(message);
        }
    }

    private void error(String message) {
        if (errors.add(message)) {
            errorCount++;
            logger.error(message);
        }
    }

}
