/*
 * Este programa es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos
 * de la licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Este programa se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA;
 * vea la licencia "GNU General Public License" para obtener mas informacion.
 */
package adalid.util.io;

import adalid.commons.properties.*;
import adalid.commons.util.*;
import adalid.util.*;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import javax.swing.filechooser.FileNameExtensionFilter;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * @author Jorge Campins
 */
public class SysCmdRunner extends Utility {

    private static final Logger logger = Logger.getLogger(Utility.class);

    public static final String MORE_INTERACTIVE = SysCmdRunner.class.getSimpleName() + "="; // unset environment variable

    public static final String LESS_INTERACTIVE = MORE_INTERACTIVE + SysCmdRunner.class.getCanonicalName(); // set environment variable

    public static void main(String[] args) {
        executeSystemCommand();
    }

    public static Process executeSystemCommand() {
        return executeSystemCommand(null);
    }

    public static Process executeSystemCommand(String folder) {
        return executeSystemCommand(folder, null);
    }

    public static Process executeSystemCommand(String folder, String command) {
        List<String> env = null;
        return executeSystemCommand(folder, command, env);
    }

    public static Process executeSystemCommand(String folder, String command, String... vars) {
        List<String> env = vars == null || vars.length == 0 ? null : Arrays.asList(vars);
        return executeSystemCommand(folder, command, env);
    }

    public static Process executeSystemCommand(String folder, String command, List<String> env) {
        File dir = directoryOf(folder);
        return executeSystemCommand(dir, commandOf(dir, command), env);
    }

    public static void executeFile() {
        executeFile(null);
    }

    public static void log(Process process) {
        if (process != null) {
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()))) {
                String line;
                while ((line = reader.readLine()) != null) {
                    logger.info(line);
                }
            } catch (IOException ex) {
                logger.fatal(ThrowableUtils.getString(ex), ex);
            }
        }
    }

    public static void send(Process process, String string) {
        if (process != null) {
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(process.getOutputStream()))) {
                writer.write(string);
            } catch (IOException ex) {
                logger.fatal(ThrowableUtils.getString(ex), ex);
            }
        }
    }

    public static void executeFile(String folder) {
        File dir = directoryOf(folder);
        String command;
        File f;
//      Process p;
        while (true) {
            command = commandOf(dir, null);
            if (command == null) {
                break;
            }
            f = new File(command);
            if (!f.isFile()) {
                break;
            }
            /*p =*/ executeSystemCommand(dir, command, null);
            /*
            for (int i = 0; i < 12; i++) {
                if (p.isAlive()) {
                    logger.info(p);
                    try {
                        p.waitFor(5, TimeUnit.SECONDS);
                    } catch (InterruptedException ex) {
                        logger.fatal(ex);
                    }
                } else {
                    break;
                }
            }
            if (p.isAlive()) {
                logger.warn(p);
            } else {
                logger.info(p);
            }
            /**/
            dir = f.getParentFile();
            sleep(3000);
        }
    }

    private static File directoryOf(String folder) {
        if (StringUtils.isBlank(folder)) {
            File rootFolder = PropertiesHandler.getRootFolder();
            logger.info("root-folder=" + rootFolder);
            folder = rootFolder.getAbsolutePath();
        }
        File dir = new File(folder);
        if (!dir.isAbsolute()) {
            File rootFolder = PropertiesHandler.getRootFolder();
            logger.info("root-folder=" + rootFolder);
            dir = new File(rootFolder, folder);
        }
        logger.info("working-dir=" + dir);
        return dir;
    }

    private static String commandOf(File dir, String command) {
        if (StringUtils.isBlank(command)) {
            command = chooseFile(dir.getAbsolutePath(), new FileNameExtensionFilter("*.bat", "bat"));
            if (StringUtils.isBlank(command)) {
                return null;
            }
        }
        return command;
    }

    private static void sleep(long millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException ex) {
            logger.fatal(ex);
        }
    }

    private static final boolean RTX = false;

    private static Process executeSystemCommand(File dir, String command, List<String> env) {
        if (dir == null || command == null) {
            return null;
        }
        SysCmdCommand scc = new SysCmdCommand(command, env, dir).forWindows();
        addLessInteractiveEnvironmentVariable(scc.env);
        try {
            return RTX ? execute(scc) : start(scc);
        } catch (IOException ex) {
            logger.fatal(ex);
        }
        return null;
    }

    private static Process execute(SysCmdCommand scc) throws IOException {
        logger.info("cmd=" + scc.command);
        logger.info("env=" + Arrays.toString(scc.env()));
        logger.info("dir=" + scc.dir);
        return Runtime.getRuntime().exec(scc.command, scc.env(), scc.dir);
    }

    private static Process start(SysCmdCommand scc) throws IOException {
        ProcessBuilder processBuilder = new ProcessBuilder(scc.command());
        if (scc.env != null && !scc.env.isEmpty()) {
            logger.trace("env=" + scc.env);
            Map<String, String> environment = processBuilder.environment();
            for (String var : scc.env) {
                String[] tokens = StringUtils.split(var, '=');
                if (tokens != null && tokens.length == 2) {
                    environment.put(tokens[0], tokens[1]);
                }
            }
        }
        processBuilder.directory(scc.dir);
        processBuilder.redirectErrorStream(true);
        logger.info("cmd=" + processBuilder.command());
        logger.info("env=" + processBuilder.environment());
        logger.info("dir=" + processBuilder.directory());
        return processBuilder.start();
    }

    /*
    private static List<String> env(String... vars) {
        if (vars == null || vars.length == 0) {
            return null;
        }
        List<String> list = new ArrayList<>();
        String key = null;
        for (String var : vars) {
            if (key == null) {
                key = var;
            } else {
                list.add(key + "=" + var);
                key = null;
            }
        }
        return list;
    }

    /**/
    private static boolean addLessInteractiveEnvironmentVariable(List<String> env) {
        if (env == null) {
            return false;
        }
        for (String var : env) {
            if (StringUtils.startsWithIgnoreCase(var, MORE_INTERACTIVE)) {
                return false;
            }
        }
        env.add(LESS_INTERACTIVE);
        return true;
    }

    public static List<String> systemEnvironmentList() {
        Map<String, String> env = System.getenv();
        List<String> list = new ArrayList<>();
        String var;
        for (String key : env.keySet()) {
            var = key + "=" + env.get(key);
            list.add(var);
        }
        return list;
    }

}
