/*
 * Copyright 2017 Jorge Campins y David Uzcategui
 *
 * Este archivo forma parte de Adalid.
 *
 * Adalid es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos de la
 * licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Adalid se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA; sin
 * siquiera las garantias implicitas de COMERCIALIZACION e IDONEIDAD PARA UN PROPOSITO PARTICULAR.
 *
 * Para mas detalles vea la licencia "GNU General Public License" en http://www.gnu.org/licenses
 */
package adalid.core.annotations;

import adalid.core.enums.*;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * La anotación EntityConsoleView se utiliza para controlar la generación de las vistas (páginas) de ejecución de operaciones de negocio de la
 * entidad, conocidas como Consolas de Procesamiento.
 *
 * @author Jorge Campins
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface EntityConsoleView {

    /**
     * enabled indica si se debe, o no, generar las vistas. Su valor es uno de los elementos de la enumeración Kleenean. Seleccione TRUE para generar
     * las vistas; en caso contrario, seleccione FALSE. Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor
     * predeterminado del atributo. El valor predeterminado del atributo es TRUE si la entidad no es una enumeración y tiene al menos una operación de
     * negocio definida; en caso contrario es FALSE.
     *
     * @return enabled
     */
    Kleenean enabled() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * menu indica si las vistas (páginas) generadas deben ser, o no, accesibles desde el menú principal de la aplicación. Su valor es uno de los
     * elementos de la enumeración Kleenean. Seleccione TRUE si las vistas deben ser accesibles desde el menú; en caso contrario, seleccione FALSE.
     * Alternativamente, omita el elemento o seleccione UNSPECIFIED para utilizar el valor predeterminado del atributo. El valor predeterminado del
     * atributo es TRUE si la entidad no es una enumeración; en caso contrario es FALSE.
     *
     * @return menu
     */
    Kleenean menu() default Kleenean.UNSPECIFIED; // TRUE

    /**
     * helpDocument especifica el documento incrustado de ayuda de las vistas (páginas) de ejecución de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, puede ser una URL o un <b>iframe</b> que incluya la URL del documento.
     *
     * Las vistas utilizarán el documento incrustado definido para el correspondiente formato de vista de la entidad, la entidad de la vista, el
     * módulo al que pertenece la entidad de la vista o el proyecto maestro, buscando en ese orden; si ninguno de ellos está definido, no habrá un
     * documento incrustado disponible para la vista.
     *
     * @return helpDocument
     */
    String helpDocument() default "";

    /**
     * helpFile especifica la ruta y el nombre del archivo de ayuda de las vistas (páginas) de ejecución de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, la ruta del archivo debe ser relativa al subdirectorio especificado mediante el atributo extraordinario
     * HELP_RESOURCES_FOLDER del proyecto maestro, y cuyo valor por omisión es el subdirectorio resources/help/custom-made del directorio
     * src/main/webapp del módulo Web de la aplicación.
     *
     * Cada vista de la entidad podría tener su propio archivo de ayuda, según lo establecido mediante el elemento helpFileAutoName de la anotación
     * EntityClass.
     *
     * La vista que no tenga su propio archivo de ayuda utilizará el definido para el correspondiente formato de vista de la entidad, la entidad de la
     * vista, el módulo al que pertenece la entidad de la vista o el proyecto maestro, buscando en ese orden; si ninguno de ellos está definido, la
     * página de ayuda no estará disponible para la vista.
     *
     * @return helpFile
     */
    String helpFile() default "";

    /**
     * headSnippet especifica la ruta y el nombre del snippet del encabezado en las vistas (páginas) de ejecución de operaciones de negocio de la
     * entidad.
     *
     * Si utiliza la plataforma jee2, los snippets se deben agregar en el subdirectorio resources/snippets/custom-made del directorio src/main/webapp
     * del módulo Web, o en algún subdirectorio de resources/snippets/custom-made; si el valor de snippet no comienza por
     * <b>/resources/snippets/custom-made/</b>, ese prefijo se le agrega automáticamente. Además, los snippets deben ser archivos xhtml; si el valor
     * de snippet no termina con <b>.xhtml</b>, ese sufijo se le agrega automáticamente.
     *
     * @return headSnippet
     */
    String headSnippet() default "";

    /**
     * easternToolbarSnippet especifica la ruta y el nombre del snippet de la barra de botones de acción oriental en las vistas (páginas) de ejecución
     * de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, los snippets se deben agregar en el subdirectorio resources/snippets/custom-made del directorio src/main/webapp
     * del módulo Web, o en algún subdirectorio de resources/snippets/custom-made; si el valor de snippet no comienza por
     * <b>/resources/snippets/custom-made/</b>, ese prefijo se le agrega automáticamente. Además, los snippets deben ser archivos xhtml; si el valor
     * de snippet no termina con <b>.xhtml</b>, ese sufijo se le agrega automáticamente.
     *
     * @return easternToolbarSnippet
     */
    String easternToolbarSnippet() default "";

    /**
     * westernToolbarSnippet especifica la ruta y el nombre del snippet de la barra de botones de acción occidental en las vistas (páginas) de
     * ejecución de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, los snippets se deben agregar en el subdirectorio resources/snippets/custom-made del directorio src/main/webapp
     * del módulo Web, o en algún subdirectorio de resources/snippets/custom-made; si el valor de snippet no comienza por
     * <b>/resources/snippets/custom-made/</b>, ese prefijo se le agrega automáticamente. Además, los snippets deben ser archivos xhtml; si el valor
     * de snippet no termina con <b>.xhtml</b>, ese sufijo se le agrega automáticamente.
     *
     * @return westernToolbarSnippet
     */
    String westernToolbarSnippet() default "";

    /**
     * aboveProcessingPanelSnippet especifica la ruta y el nombre del snippet ubicado encima del panel de procesamiento en las vistas (páginas) de
     * ejecución de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, los snippets se deben agregar en el subdirectorio resources/snippets/custom-made del directorio src/main/webapp
     * del módulo Web, o en algún subdirectorio de resources/snippets/custom-made; si el valor de snippet no comienza por
     * <b>/resources/snippets/custom-made/</b>, ese prefijo se le agrega automáticamente. Además, los snippets deben ser archivos xhtml; si el valor
     * de snippet no termina con <b>.xhtml</b>, ese sufijo se le agrega automáticamente.
     *
     * @return aboveProcessingPanelSnippet
     */
    String aboveProcessingPanelSnippet() default "";

    /**
     * belowProcessingPanelSnippet especifica la ruta y el nombre del snippet ubicado debajo del panel de procesamiento en las vistas (páginas) de
     * ejecución de operaciones de negocio de la entidad.
     *
     * Si utiliza la plataforma jee2, los snippets se deben agregar en el subdirectorio resources/snippets/custom-made del directorio src/main/webapp
     * del módulo Web, o en algún subdirectorio de resources/snippets/custom-made; si el valor de snippet no comienza por
     * <b>/resources/snippets/custom-made/</b>, ese prefijo se le agrega automáticamente. Además, los snippets deben ser archivos xhtml; si el valor
     * de snippet no termina con <b>.xhtml</b>, ese sufijo se le agrega automáticamente.
     *
     * @return belowProcessingPanelSnippet
     */
    String belowProcessingPanelSnippet() default "";

}
