/*
 * Este programa es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos
 * de la licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Este programa se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA;
 * vea la licencia "GNU General Public License" para obtener mas informacion.
 */
package showcase.meta.entidad.cliente;

import adalid.core.annotations.*;
import adalid.core.enums.*;
import adalid.core.interfaces.*;
import adalid.core.properties.*;
import java.lang.reflect.Field;
import showcase.meta.entidad.configuracion.fija.*;

@DiscriminatorValue("1")
@EntityDataGen(start = 1, step = 1, stop = 25)
@EntityDetailView(enabled = Kleenean.TRUE, menu = ViewMenuOption.WRITING, inserting = Kleenean.TRUE)
public class PersonaFisica extends Cliente {

    // <editor-fold defaultstate="collapsed" desc="class constructors">
    public PersonaFisica(Artifact declaringArtifact, Field declaringField) {
        super(declaringArtifact, declaringField);
    }
    // </editor-fold>

    @Override
    protected void addAllocationStrings() {
        super.addAllocationStrings();
        addAllocationStrings(
            "conyuge.sexo"
        );
    }

    @Override
    protected void localizeAttributes() {
        super.localizeAttributes();
        // <editor-fold defaultstate="collapsed" desc="localization of PersonaFisica's attributes">
        setLocalizedLabel(ENGLISH, "natural person");
        setLocalizedLabel(SPANISH, "persona física");
        setLocalizedCollectionLabel(ENGLISH, "Natural persons");
        setLocalizedCollectionLabel(SPANISH, "Personas Físicas");
        // </editor-fold>
    }

    @Override
    protected void localizeProperties() {
        super.localizeProperties();
        // <editor-fold defaultstate="collapsed" desc="localization of PersonaFisica's properties">
        fechaNacimiento.setLocalizedLabel(ENGLISH, "birth date");
        /**/
        edad.setLocalizedLabel(ENGLISH, "age");
        /**/
        personaAdulta.setLocalizedLabel(ENGLISH, "adult person");
        /**/
        sexo.setLocalizedLabel(ENGLISH, "sex");
        /**/
        estadoCivil.setLocalizedLabel(ENGLISH, "civil status");
        estadoCivil.setLocalizedLabel(SPANISH, "estado civil");
        /**/
        conyuge.setLocalizedLabel(ENGLISH, "spouse");
        conyuge.setLocalizedLabel(SPANISH, "cónyuge");
        /**/
        empleador.setLocalizedLabel(ENGLISH, "employer");
        // </editor-fold>
    }

    @Override
    protected void localizeExpressions() {
        super.localizeExpressions();
        // <editor-fold defaultstate="collapsed" desc="localization of PersonaFisica's expressions">
        check301.setLocalizedErrorMessage(ENGLISH, "spouse cannot be specified if the person is single");
        check301.setLocalizedErrorMessage(SPANISH, "cónyuge no se puede especificar si la persona está soltera");
        /**/
        check302.setLocalizedErrorMessage(ENGLISH, "spouse has to be someone else");
        check302.setLocalizedErrorMessage(SPANISH, "cónyuge tiene que ser otra persona");
        /**/
        check303.setLocalizedErrorMessage(ENGLISH, "spouse must be a person of the opposite sex");
        check303.setLocalizedErrorMessage(SPANISH, "cónyuge tiene que ser una persona del sexo opuesto");
        // </editor-fold>
    }

    @Override
    protected void settleAttributes() {
        super.settleAttributes();
        setDefaultLabel("persona física");
        setDefaultCollectionLabel("Personas Físicas");
        setDefaultMenuOptionLabel(DisplayFormat.DETAIL, "Agregar Personas Físicas");
        /**/
        setLocalizedMenuOptionLabel(ENGLISH, DisplayFormat.DETAIL, "Add Natural Persons");
        /**/
    }

    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyField(create = Kleenean.TRUE, table = Kleenean.FALSE, report = Kleenean.TRUE)
    @TemporalDataGen(type = DataGenType.RANDOM, min = "-60Y", max = "-21Y")
    @DateField(yearRange = 100)
    public DateProperty fechaNacimiento;

    @ColumnField(calculable = Kleenean.TRUE)
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE)
    public IntegerProperty edad;

    @ColumnField(calculable = Kleenean.TRUE)
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE)
    public BooleanProperty personaAdulta;

    @ColumnField(nullable = Kleenean.FALSE)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @PropertyField(create = Kleenean.TRUE, table = Kleenean.FALSE, report = Kleenean.TRUE)
    public SexoPersona sexo;

    @ColumnField(nullable = Kleenean.FALSE)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @PropertyField(create = Kleenean.TRUE, table = Kleenean.FALSE, report = Kleenean.TRUE)
    public EstadoCivil estadoCivil;

    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @OneToOne(navigability = Navigability.UNIDIRECTIONAL)
    @PropertyField(create = Kleenean.TRUE)
    @EntityReferenceDataGen(nullable = 100)
    @EntityReferenceSearch(displayMode = DisplayMode.WRITING)
    public PersonaFisica conyuge;

    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @PropertyField(create = Kleenean.TRUE)
    @EntityReferenceSearch(displayMode = DisplayMode.WRITING)
    public PersonaJuridica empleador;

    @Override
    protected void settleProperties() {
        super.settleProperties();
        imagen.setAvatarDefault(AvatarDefault.PERSON);
        fechaNacimiento.setMinValue("1901-01-01");
        fechaNacimiento.setMaxValue(CURRENT_DATE);
        edad.setCalculableValueExpression(CURRENT_DATE.diffInYears(fechaNacimiento));
        personaAdulta.setCalculableValueExpression(CURRENT_DATE.isGreaterOrEqualTo(fechaNacimiento.addYears(21)));
        conyuge.setDefaultLabel("cónyuge");
        empleador.setDefaultLabel("empleador");
    }

    @Override
    protected void settleLinks() {
        super.settleLinks();
        linkForeignQueryProperty(conyuge.fechaNacimiento, empleador.fechaRegistro);
    }

    @Override
    protected void settleSteps() {
        super.settleSteps();
        step001.newStepField(fechaNacimiento, edad, personaAdulta, sexo, estadoCivil, conyuge, empleador);
    }

    @Override
    protected void settleTabs() {
        super.settleTabs();
        tab001.newTabField(fechaNacimiento, edad, personaAdulta, sexo, estadoCivil, conyuge, empleador);
    }

    protected Check check301, check302, check303;

    @Override
    protected void settleExpressions() {
        super.settleExpressions();
        check301 = estadoCivil.isEqualTo(estadoCivil.SOLTERO).implies(conyuge.isNull());
        check301.setDefaultErrorMessage("cónyuge no se puede especificar si la persona está soltera");
        check302 = conyuge.isNull().or(conyuge.isNotEqualTo(this));
        check302.setDefaultErrorMessage("cónyuge tiene que ser otra persona");
        check303 = conyuge.isNull().or(conyuge.sexo.isNotEqualTo(sexo));
        check303.setDefaultErrorMessage("cónyuge tiene que ser una persona del sexo opuesto");
    }

    @Override
    protected void settleFilters() {
        super.settleFilters();
        conyuge.setSearchQueryFilter(conyuge.isNotEqualTo(this).and(conyuge.sexo.isNotEqualTo(sexo)));
    }

}
