/*
 * Este programa es software libre; usted puede redistribuirlo y/o modificarlo bajo los terminos
 * de la licencia "GNU General Public License" publicada por la Fundacion "Free Software Foundation".
 * Este programa se distribuye con la esperanza de que pueda ser util, pero SIN NINGUNA GARANTIA;
 * vea la licencia "GNU General Public License" para obtener mas informacion.
 */
package showcase.meta.entidad.pedido;

import adalid.core.*;
import adalid.core.annotations.*;
import adalid.core.enums.*;
import adalid.core.interfaces.*;
import adalid.core.page.format.*;
import adalid.core.properties.*;
import java.lang.reflect.Field;
import meta.entidad.comun.control.acceso.Usuario;
import showcase.meta.entidad.cliente.Cliente;
import showcase.meta.entidad.configuracion.Banco;
import showcase.meta.entidad.configuracion.fija.EstadoPedido;
import showcase.meta.entidad.configuracion.fija.FormaPago;
import showcase.meta.operacion.pedido.*;
import showcase.meta.propiedad.*;

@EntityCodeGen(bws = Kleenean.TRUE, fws = Kleenean.TRUE)
@EntityDataGen(start = 1, step = 1, stop = 2000)
@EntityTableView(inserts = Kleenean.FALSE, updates = Kleenean.TRUE, responsiveMode = TableResponsiveMode.AUTO)
@EntityDetailView(enabled = Kleenean.TRUE, menu = ViewMenuOption.WRITING, inserting = Kleenean.TRUE)
@EntitySelectOperation(onload = SelectOnloadOption.EXECUTE, rowsLimit = 100, sortOption = SortOption.DESC)
public class Pedido extends AbstractPersistentEntity {

    // <editor-fold defaultstate="collapsed" desc="class constructors">
    public Pedido(Artifact declaringArtifact, Field declaringField) {
        super(declaringArtifact, declaringField);
    }
    // </editor-fold>

    @Override
    protected void addAllocationStrings() {
        super.addAllocationStrings();
        super.addAllocationStrings(
            "cliente.direccionPrincipal",
            "cliente.formaPago",
            "cliente.banco"
        );
    }

    @Override
    protected void settleAttributes() {
        super.settleAttributes();
        setDefaultMenuOptionLabel(DisplayFormat.DETAIL, "Agregar Pedidos");
    }

    @PrimaryKey
    public LongProperty id;

    @VersionProperty
    public LongProperty version;

    @BusinessKey
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE)
    @CharacterDataGen(type = DataGenType.SERIES, pattern = "00000000000000000000")
    public StringProperty numero;

    @ColumnField(nullable = Kleenean.FALSE)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.TABLE_AND_DETAIL)
    @PropertyField(table = Kleenean.TRUE, report = Kleenean.TRUE, heading = Kleenean.TRUE, overlay = Kleenean.TRUE, responsivePriority = 0)
    @EntityReferenceSearch(searchType = SearchType.DISPLAY, displayMode = DisplayMode.WRITING)
    public Cliente cliente;

    @PropertyAggregation(function = AggregateFunction.MINIMUM)
    @PropertyField(create = Kleenean.TRUE, update = Kleenean.TRUE, table = Kleenean.TRUE, report = Kleenean.TRUE, heading = Kleenean.TRUE, overlay = Kleenean.TRUE, responsivePriority = 0)
    @TemporalDataGen(type = DataGenType.RANDOM, min = "-30D", max = "0")
    public DateProperty fecha;

    @ColumnField(nullable = Kleenean.FALSE)
    @PropertyAggregation(function = AggregateFunction.SUM)
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE, search = Kleenean.FALSE, table = Kleenean.TRUE, report = Kleenean.TRUE, overlay = Kleenean.TRUE, responsivePriority = 5)
    @NumericDataGen(type = DataGenType.DEFAULT)
    public PropiedadDinero monto;

    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @EntityReferenceDataGen(type = DataGenType.DEFAULT)
    public FormaPago formaPago;

    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @EntityReferenceDataGen(type = DataGenType.DEFAULT)
    @EntityReferenceSearch(searchType = SearchType.LIST, listStyle = ListStyle.NAME, displayMode = DisplayMode.WRITING)
    public Banco banco;

    @StringField(maxLength = 20)
    @CharacterDataGen(type = DataGenType.DEFAULT)
    public StringProperty cheque;

    @StringField(maxLength = 20)
    @CharacterDataGen(type = DataGenType.DEFAULT)
    public StringProperty tarjeta;

    @ColumnField(nullable = Kleenean.TRUE)
//  @PropertyField(create = Kleenean.TRUE, update = Kleenean.TRUE, table = Kleenean.FALSE, required = Kleenean.FALSE)
    @PropertyField(hidden = Kleenean.TRUE)
    @NumericDataGen(type = DataGenType.DEFAULT)
    @NumericField(type = NumericFieldType.PERCENT)
    public PropiedadPorcentaje descuento;

    @StateProperty(transitionUser = "usuarioTransicion", transitionDateTime = "fechaHoraTransicion")
    @ColumnField(nullable = Kleenean.FALSE)
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE, table = Kleenean.TRUE, report = Kleenean.TRUE, defaultCondition = DefaultCondition.IF_NULL_ON_INSERT)
    public EstadoPedido estado;

    @PropertyField(hidden = Kleenean.TRUE, defaultCondition = DefaultCondition.IF_NULL_ON_INSERT)
    public TimestampProperty fechaHoraRecibido;

    @PropertyField(hidden = Kleenean.TRUE)
    public TimestampProperty fechaHoraProcesado;

    @PropertyField(hidden = Kleenean.TRUE)
    public TimestampProperty fechaHoraCobrado;

    @PropertyField(hidden = Kleenean.TRUE)
    public TimestampProperty fechaHoraEnviado;

    @PropertyField(hidden = Kleenean.TRUE)
    public TimestampProperty fechaHoraCancelado;

    @PropertyField(hidden = Kleenean.TRUE, defaultCondition = DefaultCondition.IF_NULL_ON_INSERT)
    @TemporalDataGen(type = DataGenType.RANDOM, min = "-720h", max = "0")
    public TimestampProperty fechaHoraTransicion;

    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
//  @PropertyField(create = Kleenean.FALSE, update = Kleenean.FALSE, defaultCheckpoint = Checkpoint.WHEREVER_POSSIBLE)
    @PropertyField(hidden = Kleenean.TRUE)
    public Usuario usuarioTransicion;

    @UserProperty // esta propiedad es requerida para realizar la asunción automática after-insert
    @ForeignKey(onDelete = OnDeleteAction.NONE, onUpdate = OnUpdateAction.NONE)
    @ManyToOne(view = MasterDetailView.NONE)
    @PropertyField(hidden = Kleenean.TRUE)
    public Usuario usuarioActualizacion;

    private static final String renglonesWS = "/entity/Pedido/renglones/writingSnippet";

    @OneToMany(targetEntity = RenglonPedido.class, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REFRESH})
    @EntityCollectionField(format = DataEntryFormat.TABLE_OR_DETAIL, beforeWritingSnippet = renglonesWS, afterWritingSnippet = renglonesWS)
    public EntityCollection renglones;

    /**/
    @Override
    protected void settleProperties() {
        super.settleProperties();
        /**/
        numero.setDefaultLabel("número del pedido");
        numero.setDefaultShortLabel("número");
        numero.setInitialValue(id.toZeroPaddedString(20));
        numero.setDefaultValue(id.toZeroPaddedString(20));
        /**/
        fecha.setInitialValue(SpecialTemporalValue.CURRENT_DATE);
        fecha.setDefaultValue(SpecialTemporalValue.CURRENT_DATE);
        fecha.setMinValue(EPOCH_DATE);
        fecha.setMaxValue(CURRENT_DATE);
        /**/
        monto.setDefaultValue(0);
        monto.setInitialValue(0);
        /**/
        banco.setDefaultDescription("banco al que corresponde la cuenta o la tarjeta; solo es necesario si la forma de pago es Cheque o Tarjeta");
        cheque.setDefaultDescription("número de cheque utilizado para el pago; solo es necesario si la forma de pago es Cheque");
        tarjeta.setDefaultDescription("número de tarjeta de crédito utilizada para el pago; solo es necesaria si la forma de pago es Tarjeta");
        /**/
        descuento.setMinValue(0);
        descuento.setMaxValue(50);
        /**/
        estado.setDefaultLabel("estado del pedido");
        estado.setDefaultShortLabel("estado");
        estado.setInitialValue(estado.RECIBIDO);
        estado.setDefaultValue(estado.RECIBIDO);
        /**/
        fechaHoraRecibido.setDefaultValue(SpecialTemporalValue.CURRENT_TIMESTAMP);
        /**/
        fechaHoraTransicion.setDefaultLabel("fecha/hora última transición");
        fechaHoraTransicion.setDefaultShortLabel("fecha/hora transición");
        fechaHoraTransicion.setDefaultValue(SpecialTemporalValue.CURRENT_TIMESTAMP);
        /**/
        usuarioTransicion.setDefaultLabel("usuario última transición");
        usuarioTransicion.setDefaultShortLabel("usuario transición");
        usuarioTransicion.setDefaultValue(SpecialEntityValue.CURRENT_USER);
        /**/
    }

    @Override
    protected void settleCollections() {
        super.settleCollections();
        renglones.setDefaultLabel("renglones del pedido");
        renglones.setDefaultShortLabel("renglones");
        renglones.setDefaultDescription("colección de renglones del pedido");
        renglones.addCount(1);
//      renglones.addSum("total", bigDecimal(0.01));
    }

    protected Tab tab001, tab002;

    @Override
    protected void settleTabs() {
        super.settleTabs();
        /**/
        tab001.setDefaultLabel("Renglones del pedido");
        tab001.setDefaultShortLabel("renglones");
        tab001.newTabField(renglones);
        /**/
        tab002.setDefaultLabel("Datos del pago");
        tab002.setDefaultShortLabel("pago");
        tab002.newTabField(formaPago, banco, cheque, tarjeta, descuento);
        /*
        tab003.setDefaultLabel("Última transición");
        tab003.setDefaultShortLabel("pedido");
        tab003.newTabField(fechaHoraTransicion, usuarioTransicion);
        /**/
    }

    @Override
    protected void settleLinks() {
        super.settleLinks();
    }

    protected View v1, v2, v3;

    @Override
    protected void settleViews() {
        super.settleViews();
        /**/
        ViewField vf;
        /**/
//      v1.setPageFormatClass(DefaultPageFormat.class);
        v1.setPageFormatClass(LandscapeLegal.class);
        v1.setShareable(true);
        v1.setDefaultLabel("Pedidos por Estado y Aprobación V1");
        vf = v1.newControlField(estado.codigo);
        vf.setDefaultLabel("estado");
        v1.newControlField(cliente.aprobado);
        v1.newDetailField(numero, SortOption.ASC);
        v1.newDetailField(fecha);
        v1.newDetailField(fechaHoraTransicion);
        v1.newDetailField(cliente.nombre);
        v1.newDetailField(cliente.codigo);
        v1.newDetailField(monto, ViewFieldAggregation.SUM_COUNT_AVERAGE_DEVIATION_MINIMUM_MAXIMUM);
        /**/
//      v2.setPageFormatClass(Ledger.class);
        v2.setPageFormatClass(LandscapeLetter.class);
        v2.setShareable(true);
        v2.setDefaultLabel("Pedidos por Estado y Aprobación V2");
        vf = v2.newControlField(estado.codigo);
        vf.setDefaultLabel("estado");
        v2.newControlField(cliente.aprobado);
        v2.newDetailField(numero, SortOption.ASC);
        v2.newDetailField(fecha);
//      v2.newDetailField(fechaHoraTransicion);
        v2.newDetailField(cliente.nombre);
        v2.newDetailField(cliente.codigo);
        v2.newDetailField(monto, ViewFieldAggregation.SUM_COUNT_AVERAGE_DEVIATION_MINIMUM_MAXIMUM);
        /**/
//      v3.setPageFormatClass(Tabloid.class);
        v3.setPageFormatClass(PortraitLetter.class);
        v3.setShareable(true);
        v3.setDefaultLabel("Pedidos por Estado y Aprobación V3");
        vf = v3.newControlField(estado.codigo);
        vf.setDefaultLabel("estado");
        v3.newControlField(cliente.aprobado);
        v3.newDetailField(numero, SortOption.ASC);
        v3.newDetailField(fecha);
//      v3.newDetailField(fechaHoraTransicion);
        v3.newDetailField(cliente.nombre);
//      v3.newDetailField(cliente.codigo);
        v3.newDetailField(monto, ViewFieldAggregation.SUM_COUNT_AVERAGE_DEVIATION_MINIMUM_MAXIMUM);
        /**/
    }

//  BooleanExpression departamental, distrital;
//
    protected Segment pagoConEfectivo, pagoConCheque, pagoConTarjeta, cancelable;

    protected State recibido, procesado, cobrado, enviado, cancelado;

    /*
    protected BooleanExpression predicado;

    /**/
    @Override
    protected void settleExpressions() {
        super.settleExpressions();
        /**/
        pagoConEfectivo = formaPago.isEqualTo(formaPago.EFECTIVO);
        pagoConEfectivo.setDefaultCollectionLabel("todos los pedidos pagados con efectivo");
        pagoConEfectivo.setDefaultCollectionShortLabel("Pedidos pagados con efectivo");
        pagoConEfectivo.setDefaultDescription("el pedido fue pagado con efectivo");
        pagoConEfectivo.setDefaultErrorMessage("la forma de pago no es efectivo");
        /**/
        pagoConCheque = formaPago.isEqualTo(formaPago.CHEQUE);
        pagoConCheque.setDefaultCollectionLabel("todos los pedidos pagados con cheque");
        pagoConCheque.setDefaultCollectionShortLabel("Pedidos pagados con cheque");
        pagoConCheque.setDefaultDescription("el pedido fue pagado con cheque");
        pagoConCheque.setDefaultErrorMessage("la forma de pago no es cheque");
        /**/
        pagoConTarjeta = formaPago.isEqualTo(formaPago.TARJETA);
        pagoConTarjeta.setDefaultCollectionLabel("todos los pedidos pagados con tarjeta de crédito o débito");
        pagoConTarjeta.setDefaultCollectionShortLabel("Pedidos pagados con tarjeta");
        pagoConTarjeta.setDefaultDescription("el pedido fue pagado con tarjeta de crédito o débito");
        pagoConTarjeta.setDefaultErrorMessage("la forma de pago no es tarjeta");
        /**/
        recibido = estado.isEqualTo(estado.RECIBIDO);
        recibido.setDefaultErrorMessage("el pedido no está recibido");
        recibido.setTransitionTimestamp(fechaHoraRecibido);
        /**/
        procesado = estado.isEqualTo(estado.PROCESADO);
        procesado.setDefaultErrorMessage("el pedido no está procesado");
        procesado.setTransitionTimestamp(fechaHoraProcesado);
        /**/
        cobrado = estado.isEqualTo(estado.COBRADO);
        cobrado.setDefaultErrorMessage("el pedido no está cobrado");
        cobrado.setTransitionTimestamp(fechaHoraCobrado);
        /**/
        enviado = estado.isEqualTo(estado.ENVIADO);
        enviado.setDefaultErrorMessage("el pedido no está enviado");
        enviado.setTransitionTimestamp(fechaHoraEnviado);
        /**/
        cancelado = estado.isEqualTo(estado.CANCELADO);
        cancelado.setDefaultErrorMessage("el pedido no está cancelado");
        cancelado.setTransitionTimestamp(fechaHoraCancelado);
        /**/
        cancelable = recibido.or(procesado).or(cobrado);
        cancelable.setDefaultErrorMessage("el pedido no está recibido, procesado o cobrado");
        /**/
        setHappyPath(recibido, procesado, cobrado, enviado);
        setHappyPathDisplaySpots(HappyPathDisplaySpots.DETAIL_VIEW_AND_MASTER_HEADING);
        /*
        predicado = recibido.and(evaluate("PredicadoPedido"));
        /**/
    }

    @Override
    protected void settleFilters() {
        /**/
        addSelectSegment(pagoConEfectivo, pagoConCheque, pagoConTarjeta);
        /**/
        setUpdateFilter(recibido);
        setDeleteFilter(recibido);
        /*
        setUpdateFilter(predicado);
        setDeleteFilter(predicado);
        /**/
        tab002.setRenderingFilter(cobrado.or(cancelado));
        /**/
    }

    protected Recibir recibir;

    protected Procesar procesar;

    protected Cobrar cobrar;

    protected Enviar enviar;

    protected Cancelar cancelar;

    protected EmitirInformeDinamico informeDinamico;

    protected EmitirInformeParametrizado informeParametrizado;

    protected EmitirInformePorEstadoAprobacionV1 informePorEstadoAprobacionV1;

    protected EmitirInformePorEstadoAprobacionV2 informePorEstadoAprobacionV2;

    protected EmitirInformePorEstadoAprobacionV3 informePorEstadoAprobacionV3;

    protected ExportarArchivoPorEstadoAprobacion archivoPorEstadoAprobacion;

    @Override
    protected void settleOperations() {
        super.settleOperations();
        /**/
        insert.addTransition(null, recibido);
        procesar.addTransition(recibido, procesado);
        cobrar.addTransition(procesado, cobrado);
        enviar.addTransition(cobrado, enviado);
        /**/
        cancelar.addTransition(recibido, cancelado);
        cancelar.addTransition(procesado, cancelado);
        cancelar.addTransition(cobrado, cancelado);
        /**/
        procesar.addTriggerOn(recibido);
        cobrar.addTriggerOn(procesado);
        enviar.addTriggerOn(cobrado);
        /**/
    }

}
