
package com.opencorporates.schemas;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;


/**
 * A Register Entry represents an entry in an official register, such as a charity/nonprofit register, or register of lobbyists, for example. Some registers -- such as the Charity Register for England & Wales have a statutory purpose, and act as a record of effectively licensing an entity to operate as a charity (as do, for example, some banking registers). Others, such as the UK's voluntary lobbying register, or the Canadian register of exporters, are more collections of entities and their activities. In general we are using register entries for those registers where there is no good normalised useful model; but where we have something such as the FCA's register of financial advisers etc, which is essentially a record of licences issued to operate in financial markets, we recommend modelling as licences.
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
    "subject_entity",
    "register",
    "identifier",
    "identifier_system_code",
    "start_date",
    "end_date",
    "source_url",
    "register_url",
    "confidence",
    "sample_date",
    "retrieved_at",
    "status",
    "category",
    "other_attributes"
})
public class OCRegisterEntrySchema {

    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("subject_entity")
    @Valid
    @NotNull
    private OCEntity subjectEntity;
    /**
     * Register
     * <p>
     * The official register of which this entry is a part
     * (Required)
     * 
     */
    @JsonProperty("register")
    @JsonPropertyDescription("The official register of which this entry is a part")
    @Valid
    @NotNull
    private OCRegister register;
    /**
     * A unique identifier used by the register to identifier the register entry. In some cases – e.g. Charity Register for England & Wales, these identifiers (in this case known as the 'charity number') are used outside of the context of the register
     * 
     */
    @JsonProperty("identifier")
    @JsonPropertyDescription("A unique identifier used by the register to identifier the register entry. In some cases \u2013 e.g. Charity Register for England & Wales, these identifiers (in this case known as the 'charity number') are used outside of the context of the register")
    private String identifier;
    /**
     * A code representing the identifier scheme. Some examples of this are us_fein (US Federal tax number), us_sec_cik (US Securities and Exchange Commission CIK), uk_ew_cc (Charity Commission of England & Wales), lei (Global Legal Entity Identifier System). Not all registers have identifier_systems, and some registers may use identifiers from other identifiers systems (e.g. the EBA is planning on using LEI in its bank register)
     * 
     */
    @JsonProperty("identifier_system_code")
    @JsonPropertyDescription("A code representing the identifier scheme. Some examples of this are us_fein (US Federal tax number), us_sec_cik (US Securities and Exchange Commission CIK), uk_ew_cc (Charity Commission of England & Wales), lei (Global Legal Entity Identifier System). Not all registers have identifier_systems, and some registers may use identifiers from other identifiers systems (e.g. the EBA is planning on using LEI in its bank register)")
    private String identifierSystemCode;
    /**
     * The date the register entry started, e.g. the date the entity was registered from, or appeared on the register the first time
     * 
     */
    @JsonProperty("start_date")
    @JsonPropertyDescription("The date the register entry started, e.g. the date the entity was registered from, or appeared on the register the first time")
    private LocalDate startDate;
    /**
     * The date the register entry ended, e.g. the date the charity was deregistered from
     * 
     */
    @JsonProperty("end_date")
    @JsonPropertyDescription("The date the register entry ended, e.g. the date the charity was deregistered from")
    private LocalDate endDate;
    /**
     * Source of the data – this may be the url of the entity on the register, the URL of an open data dump, or if there is no permanent URLs, the main URL for the register
     * (Required)
     * 
     */
    @JsonProperty("source_url")
    @JsonPropertyDescription("Source of the data \u2013 this may be the url of the entity on the register, the URL of an open data dump, or if there is no permanent URLs, the main URL for the register")
    @Size(min = 8)
    @NotNull
    private String sourceUrl;
    /**
     * URL of the entity on the register, if there is a permanent URL
     * 
     */
    @JsonProperty("register_url")
    @JsonPropertyDescription("URL of the entity on the register, if there is a permanent URL")
    @Size(min = 8)
    private String registerUrl;
    /**
     * Confidence in accuracy of data
     * 
     */
    @JsonProperty("confidence")
    @JsonPropertyDescription("Confidence in accuracy of data")
    private OCRegisterEntrySchema.Confidence confidence;
    /**
     * Date on which we know this to be true (usually date this information was retrieved from the source)
     * (Required)
     * 
     */
    @JsonProperty("sample_date")
    @JsonPropertyDescription("Date on which we know this to be true (usually date this information was retrieved from the source)")
    @NotNull
    private LocalDate sampleDate;
    /**
     * The time or date at which the source URL was requested
     * (Required)
     * 
     */
    @JsonProperty("retrieved_at")
    @JsonPropertyDescription("The time or date at which the source URL was requested")
    @NotNull
    private LocalDate retrievedAt;
    /**
     * The status of the register entry
     * 
     */
    @JsonProperty("status")
    @JsonPropertyDescription("The status of the register entry")
    private String status;
    /**
     * Category of register entry (NB some registers may have more than one type of entry, so we record at the register entry level, rather than the register level)
     * (Required)
     * 
     */
    @JsonProperty("category")
    @JsonPropertyDescription("Category of register entry (NB some registers may have more than one type of entry, so we record at the register entry level, rather than the register level)")
    @NotNull
    private OCRegisterEntrySchema.Category category;
    /**
     * Use for other licence attributes for which we don't yet have curated schema attributes
     * 
     */
    @JsonProperty("other_attributes")
    @JsonPropertyDescription("Use for other licence attributes for which we don't yet have curated schema attributes")
    @Valid
    private OCOtherAttributes__6 otherAttributes;

    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("subject_entity")
    public OCEntity getSubjectEntity() {
        return subjectEntity;
    }

    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("subject_entity")
    public void setSubjectEntity(OCEntity subjectEntity) {
        this.subjectEntity = subjectEntity;
    }

    /**
     * Register
     * <p>
     * The official register of which this entry is a part
     * (Required)
     * 
     */
    @JsonProperty("register")
    public OCRegister getRegister() {
        return register;
    }

    /**
     * Register
     * <p>
     * The official register of which this entry is a part
     * (Required)
     * 
     */
    @JsonProperty("register")
    public void setRegister(OCRegister register) {
        this.register = register;
    }

    /**
     * A unique identifier used by the register to identifier the register entry. In some cases – e.g. Charity Register for England & Wales, these identifiers (in this case known as the 'charity number') are used outside of the context of the register
     * 
     */
    @JsonProperty("identifier")
    public String getIdentifier() {
        return identifier;
    }

    /**
     * A unique identifier used by the register to identifier the register entry. In some cases – e.g. Charity Register for England & Wales, these identifiers (in this case known as the 'charity number') are used outside of the context of the register
     * 
     */
    @JsonProperty("identifier")
    public void setIdentifier(String identifier) {
        this.identifier = identifier;
    }

    /**
     * A code representing the identifier scheme. Some examples of this are us_fein (US Federal tax number), us_sec_cik (US Securities and Exchange Commission CIK), uk_ew_cc (Charity Commission of England & Wales), lei (Global Legal Entity Identifier System). Not all registers have identifier_systems, and some registers may use identifiers from other identifiers systems (e.g. the EBA is planning on using LEI in its bank register)
     * 
     */
    @JsonProperty("identifier_system_code")
    public String getIdentifierSystemCode() {
        return identifierSystemCode;
    }

    /**
     * A code representing the identifier scheme. Some examples of this are us_fein (US Federal tax number), us_sec_cik (US Securities and Exchange Commission CIK), uk_ew_cc (Charity Commission of England & Wales), lei (Global Legal Entity Identifier System). Not all registers have identifier_systems, and some registers may use identifiers from other identifiers systems (e.g. the EBA is planning on using LEI in its bank register)
     * 
     */
    @JsonProperty("identifier_system_code")
    public void setIdentifierSystemCode(String identifierSystemCode) {
        this.identifierSystemCode = identifierSystemCode;
    }

    /**
     * The date the register entry started, e.g. the date the entity was registered from, or appeared on the register the first time
     * 
     */
    @JsonProperty("start_date")
    public LocalDate getStartDate() {
        return startDate;
    }

    /**
     * The date the register entry started, e.g. the date the entity was registered from, or appeared on the register the first time
     * 
     */
    @JsonProperty("start_date")
    public void setStartDate(LocalDate startDate) {
        this.startDate = startDate;
    }

    /**
     * The date the register entry ended, e.g. the date the charity was deregistered from
     * 
     */
    @JsonProperty("end_date")
    public LocalDate getEndDate() {
        return endDate;
    }

    /**
     * The date the register entry ended, e.g. the date the charity was deregistered from
     * 
     */
    @JsonProperty("end_date")
    public void setEndDate(LocalDate endDate) {
        this.endDate = endDate;
    }

    /**
     * Source of the data – this may be the url of the entity on the register, the URL of an open data dump, or if there is no permanent URLs, the main URL for the register
     * (Required)
     * 
     */
    @JsonProperty("source_url")
    public String getSourceUrl() {
        return sourceUrl;
    }

    /**
     * Source of the data – this may be the url of the entity on the register, the URL of an open data dump, or if there is no permanent URLs, the main URL for the register
     * (Required)
     * 
     */
    @JsonProperty("source_url")
    public void setSourceUrl(String sourceUrl) {
        this.sourceUrl = sourceUrl;
    }

    /**
     * URL of the entity on the register, if there is a permanent URL
     * 
     */
    @JsonProperty("register_url")
    public String getRegisterUrl() {
        return registerUrl;
    }

    /**
     * URL of the entity on the register, if there is a permanent URL
     * 
     */
    @JsonProperty("register_url")
    public void setRegisterUrl(String registerUrl) {
        this.registerUrl = registerUrl;
    }

    /**
     * Confidence in accuracy of data
     * 
     */
    @JsonProperty("confidence")
    public OCRegisterEntrySchema.Confidence getConfidence() {
        return confidence;
    }

    /**
     * Confidence in accuracy of data
     * 
     */
    @JsonProperty("confidence")
    public void setConfidence(OCRegisterEntrySchema.Confidence confidence) {
        this.confidence = confidence;
    }

    /**
     * Date on which we know this to be true (usually date this information was retrieved from the source)
     * (Required)
     * 
     */
    @JsonProperty("sample_date")
    public LocalDate getSampleDate() {
        return sampleDate;
    }

    /**
     * Date on which we know this to be true (usually date this information was retrieved from the source)
     * (Required)
     * 
     */
    @JsonProperty("sample_date")
    public void setSampleDate(LocalDate sampleDate) {
        this.sampleDate = sampleDate;
    }

    /**
     * The time or date at which the source URL was requested
     * (Required)
     * 
     */
    @JsonProperty("retrieved_at")
    public LocalDate getRetrievedAt() {
        return retrievedAt;
    }

    /**
     * The time or date at which the source URL was requested
     * (Required)
     * 
     */
    @JsonProperty("retrieved_at")
    public void setRetrievedAt(LocalDate retrievedAt) {
        this.retrievedAt = retrievedAt;
    }

    /**
     * The status of the register entry
     * 
     */
    @JsonProperty("status")
    public String getStatus() {
        return status;
    }

    /**
     * The status of the register entry
     * 
     */
    @JsonProperty("status")
    public void setStatus(String status) {
        this.status = status;
    }

    /**
     * Category of register entry (NB some registers may have more than one type of entry, so we record at the register entry level, rather than the register level)
     * (Required)
     * 
     */
    @JsonProperty("category")
    public OCRegisterEntrySchema.Category getCategory() {
        return category;
    }

    /**
     * Category of register entry (NB some registers may have more than one type of entry, so we record at the register entry level, rather than the register level)
     * (Required)
     * 
     */
    @JsonProperty("category")
    public void setCategory(OCRegisterEntrySchema.Category category) {
        this.category = category;
    }

    /**
     * Use for other licence attributes for which we don't yet have curated schema attributes
     * 
     */
    @JsonProperty("other_attributes")
    public OCOtherAttributes__6 getOtherAttributes() {
        return otherAttributes;
    }

    /**
     * Use for other licence attributes for which we don't yet have curated schema attributes
     * 
     */
    @JsonProperty("other_attributes")
    public void setOtherAttributes(OCOtherAttributes__6 otherAttributes) {
        this.otherAttributes = otherAttributes;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(OCRegisterEntrySchema.class.getName()).append('@').append(Integer.toHexString(System.identityHashCode(this))).append('[');
        sb.append("subjectEntity");
        sb.append('=');
        sb.append(((this.subjectEntity == null)?"<null>":this.subjectEntity));
        sb.append(',');
        sb.append("register");
        sb.append('=');
        sb.append(((this.register == null)?"<null>":this.register));
        sb.append(',');
        sb.append("identifier");
        sb.append('=');
        sb.append(((this.identifier == null)?"<null>":this.identifier));
        sb.append(',');
        sb.append("identifierSystemCode");
        sb.append('=');
        sb.append(((this.identifierSystemCode == null)?"<null>":this.identifierSystemCode));
        sb.append(',');
        sb.append("startDate");
        sb.append('=');
        sb.append(((this.startDate == null)?"<null>":this.startDate));
        sb.append(',');
        sb.append("endDate");
        sb.append('=');
        sb.append(((this.endDate == null)?"<null>":this.endDate));
        sb.append(',');
        sb.append("sourceUrl");
        sb.append('=');
        sb.append(((this.sourceUrl == null)?"<null>":this.sourceUrl));
        sb.append(',');
        sb.append("registerUrl");
        sb.append('=');
        sb.append(((this.registerUrl == null)?"<null>":this.registerUrl));
        sb.append(',');
        sb.append("confidence");
        sb.append('=');
        sb.append(((this.confidence == null)?"<null>":this.confidence));
        sb.append(',');
        sb.append("sampleDate");
        sb.append('=');
        sb.append(((this.sampleDate == null)?"<null>":this.sampleDate));
        sb.append(',');
        sb.append("retrievedAt");
        sb.append('=');
        sb.append(((this.retrievedAt == null)?"<null>":this.retrievedAt));
        sb.append(',');
        sb.append("status");
        sb.append('=');
        sb.append(((this.status == null)?"<null>":this.status));
        sb.append(',');
        sb.append("category");
        sb.append('=');
        sb.append(((this.category == null)?"<null>":this.category));
        sb.append(',');
        sb.append("otherAttributes");
        sb.append('=');
        sb.append(((this.otherAttributes == null)?"<null>":this.otherAttributes));
        sb.append(',');
        if (sb.charAt((sb.length()- 1)) == ',') {
            sb.setCharAt((sb.length()- 1), ']');
        } else {
            sb.append(']');
        }
        return sb.toString();
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = ((result* 31)+((this.identifier == null)? 0 :this.identifier.hashCode()));
        result = ((result* 31)+((this.subjectEntity == null)? 0 :this.subjectEntity.hashCode()));
        result = ((result* 31)+((this.endDate == null)? 0 :this.endDate.hashCode()));
        result = ((result* 31)+((this.confidence == null)? 0 :this.confidence.hashCode()));
        result = ((result* 31)+((this.otherAttributes == null)? 0 :this.otherAttributes.hashCode()));
        result = ((result* 31)+((this.sourceUrl == null)? 0 :this.sourceUrl.hashCode()));
        result = ((result* 31)+((this.identifierSystemCode == null)? 0 :this.identifierSystemCode.hashCode()));
        result = ((result* 31)+((this.registerUrl == null)? 0 :this.registerUrl.hashCode()));
        result = ((result* 31)+((this.retrievedAt == null)? 0 :this.retrievedAt.hashCode()));
        result = ((result* 31)+((this.sampleDate == null)? 0 :this.sampleDate.hashCode()));
        result = ((result* 31)+((this.category == null)? 0 :this.category.hashCode()));
        result = ((result* 31)+((this.startDate == null)? 0 :this.startDate.hashCode()));
        result = ((result* 31)+((this.register == null)? 0 :this.register.hashCode()));
        result = ((result* 31)+((this.status == null)? 0 :this.status.hashCode()));
        return result;
    }

    @Override
    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if ((other instanceof OCRegisterEntrySchema) == false) {
            return false;
        }
        OCRegisterEntrySchema rhs = ((OCRegisterEntrySchema) other);
        return (((((((((((((((this.identifier == rhs.identifier)||((this.identifier!= null)&&this.identifier.equals(rhs.identifier)))&&((this.subjectEntity == rhs.subjectEntity)||((this.subjectEntity!= null)&&this.subjectEntity.equals(rhs.subjectEntity))))&&((this.endDate == rhs.endDate)||((this.endDate!= null)&&this.endDate.equals(rhs.endDate))))&&((this.confidence == rhs.confidence)||((this.confidence!= null)&&this.confidence.equals(rhs.confidence))))&&((this.otherAttributes == rhs.otherAttributes)||((this.otherAttributes!= null)&&this.otherAttributes.equals(rhs.otherAttributes))))&&((this.sourceUrl == rhs.sourceUrl)||((this.sourceUrl!= null)&&this.sourceUrl.equals(rhs.sourceUrl))))&&((this.identifierSystemCode == rhs.identifierSystemCode)||((this.identifierSystemCode!= null)&&this.identifierSystemCode.equals(rhs.identifierSystemCode))))&&((this.registerUrl == rhs.registerUrl)||((this.registerUrl!= null)&&this.registerUrl.equals(rhs.registerUrl))))&&((this.retrievedAt == rhs.retrievedAt)||((this.retrievedAt!= null)&&this.retrievedAt.equals(rhs.retrievedAt))))&&((this.sampleDate == rhs.sampleDate)||((this.sampleDate!= null)&&this.sampleDate.equals(rhs.sampleDate))))&&((this.category == rhs.category)||((this.category!= null)&&this.category.equals(rhs.category))))&&((this.startDate == rhs.startDate)||((this.startDate!= null)&&this.startDate.equals(rhs.startDate))))&&((this.register == rhs.register)||((this.register!= null)&&this.register.equals(rhs.register))))&&((this.status == rhs.status)||((this.status!= null)&&this.status.equals(rhs.status))));
    }


    /**
     * Category of register entry (NB some registers may have more than one type of entry, so we record at the register entry level, rather than the register level)
     * 
     */
    public enum Category {

        CHARITY_NON_PROFIT("Charity-NonProfit"),
        LOBBYING("Lobbying"),
        LEGAL_ENTITY("LegalEntity"),
        OTHER("Other");
        private final String value;
        private final static Map<String, OCRegisterEntrySchema.Category> CONSTANTS = new HashMap<String, OCRegisterEntrySchema.Category>();

        static {
            for (OCRegisterEntrySchema.Category c: values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private Category(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

        @JsonCreator
        public static OCRegisterEntrySchema.Category fromValue(String value) {
            OCRegisterEntrySchema.Category constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

    }


    /**
     * Confidence in accuracy of data
     * 
     */
    public enum Confidence {

        HIGH("HIGH"),
        MEDIUM("MEDIUM"),
        LOW("LOW");
        private final String value;
        private final static Map<String, OCRegisterEntrySchema.Confidence> CONSTANTS = new HashMap<String, OCRegisterEntrySchema.Confidence>();

        static {
            for (OCRegisterEntrySchema.Confidence c: values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private Confidence(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

        @JsonCreator
        public static OCRegisterEntrySchema.Confidence fromValue(String value) {
            OCRegisterEntrySchema.Confidence constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

    }

}
