package org.qianalyze.maven;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.drools.builder.KnowledgeBuilder;
import org.drools.builder.KnowledgeBuilderConfiguration;
import org.drools.builder.KnowledgeBuilderFactory;
import org.drools.builder.ResourceType;
import org.drools.compiler.DroolsParserException;
import org.drools.core.util.DroolsStreamUtils;
import org.drools.decisiontable.InputType;
import org.drools.decisiontable.SpreadsheetCompiler;
import org.drools.definition.KnowledgePackage;
import org.drools.definition.rule.Rule;
import org.drools.definitions.impl.KnowledgePackageImp;
import org.drools.io.ResourceFactory;

/**
 * Goal which Compiles drools rules into a package.
 * 
 * @goal compile
 * @phase compile
 * @requiresDependencyResolution compile+runtime
 */
@Mojo(name = "compile", defaultPhase = LifecyclePhase.COMPILE)
public class DroolsRulesMojo extends AbstractMojo {

    private static final String DEFAULT_INCLUDES = "**/*.drl,**/*.xls,**/*.rf";

    /**
     * Location of the file.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private File outputDirectory;

    /**
     * @parameter expression="${project.basedir}"
     * @required
     */
    private File sourceDir;

//    /**
//     * @parameter expression="${project.artifactId}"
//     * @required
//     */
//    private String artifactID;

    /**
     * @parameter
     */
    private List<String> excludes;

    /**
     * @parameter
     */
    private List<String> includes;

    /**
     * 
     * @parameter default-value="${project.compileClasspathElements}"
     * @required
     * @readonly
     */
    protected List<String> compileClasspathElements;

    /**
     * Setup the options to pass to the code generator.
     * 
     * @return the configured options.
     */
    public void execute() throws MojoExecutionException, MojoFailureException {
        try {
            System.out.println(outputDirectory.getAbsolutePath());

            compileRules();
        } catch (Exception e) {
            // Normally don't log when an exception is thrown.
            // This just helps debugging more quickly for rules errors.
            getLog().error(e);
            throw new RuntimeException(e);
        }

    }

    public void compileRules() throws DroolsParserException, IOException {

        // This is the target directory for where the rules will be placed during
        // the build.
        String targetPath = outputDirectory.toString() + "/classes/";
        File target = new File(targetPath);
        if (!target.exists()) {
            target.mkdirs();
        }

        URL modelJarURL = new URL("file", "localhost", targetPath);
        List<URL> urlList = new ArrayList<URL>();
        urlList.add(modelJarURL);

        // This is a hack to solve the problem of the drools compiler not finding the Equals class.
        List<String> compileClasspathElements = this.compileClasspathElements;
        for (String compileSourccompileClasspathElement : compileClasspathElements) {
            if (compileSourccompileClasspathElement.contains("jaxb2-basics-runtime")) {
                File compileSourceDirectory = new File(compileSourccompileClasspathElement);
                if (!compileSourceDirectory.isDirectory()) {
                    getLog().debug("Add to projectRealm: " + compileSourceDirectory.toURI().toURL().toString());
                    urlList.add(compileSourceDirectory.toURI().toURL());
                }
            }
        }
        // ClassLoader classLoader = this.getClass().getClassLoader();

        URLClassLoader classLoader = new URLClassLoader(urlList.toArray(new URL[0]));

        KnowledgeBuilderConfiguration kbuilderConf = KnowledgeBuilderFactory.newKnowledgeBuilderConfiguration(null,
                classLoader, this.getClass().getClassLoader());
        KnowledgeBuilder kbuilder = KnowledgeBuilderFactory.newKnowledgeBuilder(kbuilderConf);

        // Depending on how the project is checked out or the maven pom
        // sometimes the getDirectory() will not have the name of the project/pom
        // but unnamed since it cannot determine the correct name.
        if (StringUtils.contains(targetPath, "Unnamed")) {
            throw new RuntimeException("The project was incorrectly checked out so that maven cannot determine the"
                    + " project name, therefore it creates a directory with Unnamed.  Fix the project or add the name"
                    + " property into the Maven pom.xml");
        }

        // Ask the mavenProject for all it's resource locations.
        getLog().debug("Target directory to write output file - " + targetPath);

        // Loop through the collection, even though we really could lock this down
        // to just be the /src/main/resources. We could also change the plugin
        // to differentiate between running against test rules and production
        // rules.

        getLog().debug("Resource directory to pull rules from - " + sourceDir.getCanonicalPath());

        // The following method call allows us to use the capabilities of
        // wildcarding like we see in most maven plugins. Where we can do
        // things such as **/*.xls. This call will also resolve any conflicting
        // includes and excludes.

        List<File> listOfFiles = org.codehaus.plexus.util.FileUtils.getFiles(sourceDir, getIncludesList(),
                getExcludesList(), true);

        if (CollectionUtils.isEmpty(listOfFiles)) {
            getLog().warn("No files were found to compile into package");
        } else {
            for (File file : listOfFiles) {

                if (file.isFile()) {
                    String fileName = file.getName();

                    // Store the suffix so we can determine the rules type
                    String fileSuffix = StringUtils.substringAfterLast(fileName, ".");

                    if (StringUtils.endsWith(fileSuffix, "drl")) {
                        getLog().info("- Adding DRL: " + fileName + " to package");
                        kbuilder.add(ResourceFactory.newFileResource(file.getAbsolutePath()), ResourceType.DRL);
                    } else if (StringUtils.endsWith(fileSuffix, "xls")) {
                        // Decision tables have an extra step, so we provide additional
                        // output information.
                        SpreadsheetCompiler sc = new SpreadsheetCompiler();
                        String drlstr = sc.compile(ResourceFactory.newFileResource(file.getAbsolutePath())
                                .getInputStream(), InputType.XLS);
                        getLog().debug("--- XLS compile details = " + drlstr);
                        getLog().info("- Adding Decision Table: " + fileName + " to package");
                        kbuilder.add(ResourceFactory.newFileResource(file.getAbsolutePath()), ResourceType.DTABLE);
                    } else if (StringUtils.endsWith(fileSuffix, "rf")) {
                        getLog().info("- Adding Rule Flow: " + fileName + " to package");
                        kbuilder.add(ResourceFactory.newFileResource(file.getAbsolutePath()), ResourceType.DRF);
                    }

                    fileName = fileName.replace(fileSuffix, "pkg");

                    // If any errors have occurred in the KnowledgeBuilder, then throw
                    // the errors and stop the plugin process.
                    if (kbuilder.hasErrors()) {
                        getLog().error("Errors " + kbuilder.getErrors().toString());
                        throw new RuntimeException(kbuilder.getErrors().toString());
                    }

                    int pkgCount = 0;
                    Collection<KnowledgePackage> kpkgs = kbuilder.getKnowledgePackages();
                    for (KnowledgePackage kpkg : kpkgs) {
                        Collection<Rule> rules = kpkg.getRules();
                        for (Object element : rules) {
                            org.drools.definitions.rule.impl.RuleImpl rule = (org.drools.definitions.rule.impl.RuleImpl) element;

                            // Providing additional debug information for rules within the KnowledgeBuilder
                            getLog().debug(
                                    "<RuleName> : <Activation-Group (null=non-existent)> : <Agenda-group>:   "
                                            + rule.getName() + " : " + rule.getRule().getActivationGroup() + " : "
                                            + rule.getRule().getRuleFlowGroup());
                        }

                        if (++pkgCount > 1) {
                            getLog().warn(
                                    "Your rules are being built into multiple packages, this is most likely an issue.");
                        }

                        // Create the pkg on the file system.
                        FileOutputStream out = null;

                        try {
                            File targetFile = new File(targetPath);
                            if (!targetFile.exists()) {
                                FileUtils.forceMkdir(new File(targetPath));
                            }
                            File outFilePath = new File(targetPath + fileName );
                            out = new FileOutputStream(outFilePath);
                            DroolsStreamUtils.streamOut(out, ((KnowledgePackageImp) kpkg).pkg);
                            long filesize = outFilePath.length();
                            long filesizeInKB = filesize / 1024;

                            // Log information about the create package.
                            getLog().info(
                                    "- Package " + fileName + " built, putting pkg: " + targetPath + fileName );

                            getLog().info("- Size of " + fileName + " is: " + filesizeInKB + " KB");
                        } catch (IOException ex) {
                            throw new RuntimeException(ex);
                        } finally {
                            IOUtils.closeQuietly(out);
                        }
                    }
                }
            }
        }
    }

    /**
     * Create the exclude list. If the maven plugin was configured with excludes, then convert the list into a Csv since
     * the Plexus utility needs a comma separated list, else return a null String.
     * 
     * @return String containing the list of excludes.
     */
    private String getExcludesList() {
        if (CollectionUtils.isNotEmpty(excludes)) {
            return convertListToCsv(excludes);
        } else {
            return null;
        }
    }

    /**
     * Create the include list. If the maven plugin was configured with includes, then convert the list into a Csv since
     * the Plexus utility needs a comma separated list, else return the list of default rules types that will be
     * compiled.
     * 
     * @return String containing the list of excludes.
     */
    private String getIncludesList() {
        if (CollectionUtils.isNotEmpty(includes)) {
            return convertListToCsv(includes);

        } else {
            return DEFAULT_INCLUDES;
        }
    }

    /**
     * Looks through a list of Strings and creates a comma separated String.
     * 
     * @param includeList
     *        List of String values for include/exclude.
     * @return String that is a Csv of values.
     */
    private String convertListToCsv(List<String> includeList) {
        StringBuilder nameBuilder = new StringBuilder();
        for (String includeItem : includeList) {
            nameBuilder.append(includeItem).append(",");
        }

        nameBuilder.deleteCharAt(nameBuilder.length() - 1);
        return nameBuilder.toString();
    }
}
