/*
 *  Copyright 2024-2025 NetCracker Technology Corporation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.qubership.automation.itf.ui.swagger.plugins;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static springfox.documentation.service.Tags.emptyTags;

import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.OperationBuilderPlugin;
import springfox.documentation.spi.service.contexts.OperationContext;
import springfox.documentation.spring.web.readers.operation.DefaultTagsProvider;
import springfox.documentation.swagger.common.SwaggerPluginSupport;

/**
 * Automatic generation of controller tags
 * Source: https://github.com/springfox/springfox/issues/2134
 */
@Component
@Order(SwaggerPluginSupport.SWAGGER_PLUGIN_ORDER + 10) // later execution priority than SwaggerOperationTagsReader.
@ConditionalOnProperty(name = "swagger.enabled", havingValue = "true")
public class DisableAutoGeneratedTagsPlugin implements OperationBuilderPlugin {

    private final DefaultTagsProvider tagsProvider;

    @Autowired
    public DisableAutoGeneratedTagsPlugin(DefaultTagsProvider tagsProvider) {
        this.tagsProvider = tagsProvider;
    }

    @Override
    public void apply(OperationContext context) {
        Set<String> defaultTags = tagsProvider.tags(context);
        Set<String> tagsOnController = controllerTags(context);
        Set<String> tagsOnOperation = operationTags(context);
        SetView<String> intersectionTags = Sets.intersection(tagsOnController, tagsOnOperation);
        SetView<String> allTags = Sets.union(tagsOnController, tagsOnOperation);
        if (allTags.isEmpty()) {
            context.operationBuilder().tags(defaultTags);
        } else if (intersectionTags.isEmpty()) {
            context.operationBuilder().tags(allTags);
        } else {
            context.operationBuilder().tags(tagsOnOperation);
        }
    }

    @Override
    public boolean supports(DocumentationType delimiter) {
        return true;
    }

    private Set<String> controllerTags(OperationContext context) {
        Optional<Api> controllerAnnotation = context.findControllerAnnotation(Api.class);
        return controllerAnnotation.transform(tagsFromController()).or(Sets.newHashSet());
    }

    private Set<String> operationTags(OperationContext context) {
        Optional<ApiOperation> annotation = context.findAnnotation(ApiOperation.class);
        return annotation.transform(tagsFromOperation()).or(Sets.newHashSet());
    }

    private Function<ApiOperation, Set<String>> tagsFromOperation() {
        return input -> {
            Set<String> tags = Sets.newTreeSet();
            if (input != null) {
                tags.addAll(from(newArrayList(input.tags())).filter(emptyTags()).toSet());
            }
            return tags;
        };
    }

    private Function<Api, Set<String>> tagsFromController() {
        return input -> {
            Set<String> tags = Sets.newTreeSet();
            if (input != null) {
                tags.addAll(from(newArrayList(input.tags())).filter(emptyTags()).toSet());
            }
            return tags;
        };
    }
}
