/*
 * # Copyright 2024-2025 NetCracker Technology Corporation
 * #
 * # Licensed under the Apache License, Version 2.0 (the "License");
 * # you may not use this file except in compliance with the License.
 * # You may obtain a copy of the License at
 * #
 * #      http://www.apache.org/licenses/LICENSE-2.0
 * #
 * # Unless required by applicable law or agreed to in writing, software
 * # distributed under the License is distributed on an "AS IS" BASIS,
 * # WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * # See the License for the specific language governing permissions and
 * # limitations under the License.
 */

package org.qubership.atp.environments.utils.cloud;

import java.util.List;
import java.util.UUID;

import org.qubership.atp.environments.errorhandling.clients.EnvironmentCloudClientCreationException;
import org.qubership.atp.environments.model.ConnectionParameters;
import org.qubership.atp.environments.model.System;
import org.qubership.atp.environments.service.rest.server.response.ShortExternalService;
import org.qubership.atp.environments.utils.cloud.model.CloudService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

public abstract class ExternalCloudClient {

    protected String serverUrl;
    protected String namespace;
    private static final Logger log = LoggerFactory.getLogger(KubeClient.class);

    public abstract List<ShortExternalService> getShortServices();

    /**
     * Create cloud client by client class.
     *
     * @param connectionParameters - connection parameters for creating client.
     * @param clientClass          - class of required cloud client.
     * @return Cloud client
     */
    @SuppressFBWarnings("REC_CATCH_EXCEPTION")
    public static ExternalCloudClient createClient(ConnectionParameters connectionParameters,
                                                   Class<? extends ExternalCloudClient> clientClass) {
        ExternalCloudClient client;
        try {
            client = clientClass.getConstructor().newInstance();
            client.configure(connectionParameters);
        } catch (Exception e) {
            String clientClassName = clientClass.getName();
            log.error("Failed to create cloud client for class: {}", clientClassName);
            throw new EnvironmentCloudClientCreationException(clientClassName);
        }
        return client;
    }

    public abstract List<CloudService> getServicesByServiceId(List<UUID> ids);

    public abstract List<CloudService> getServicesByExistingServices(List<System> services);

    public abstract void configure(ConnectionParameters connectionParameters);
}
