/*
 *  Copyright 2024-2025 NetCracker Technology Corporation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.qubership.atp.mia.ei.component;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.qubership.atp.mia.ei.service.AtpImportStrategy;
import org.qubership.atp.mia.model.configuration.ProjectConfiguration;
import org.qubership.atp.mia.model.file.ProjectDirectory;
import org.qubership.atp.mia.model.file.ProjectFile;

public class ExportImportUtils {

    /**
     * Get sourceId of {@link ProjectDirectory}, if not present gets id.
     *
     * @param directory - directory to get id from.
     * @return sourceId or id or null.
     */
    public static UUID getFolderId(ProjectDirectory directory) {
        return isNull(directory) ? null : Optional.ofNullable(directory.getSourceId()).orElse(directory.getId());
    }

    /**
     * In case of file name conflicts this method provide unique name for existing file.
     * e.g. File1.txt -> File1.txt_ConflictImport_1732861570280
     *
     * @param name - current name to modify.
     * @return modified name with some suffix and current UNIX timestamp.
     */
    public static String generateCopyEntryName(String name) {
        if (isNull(name) || name.isEmpty()) {
            name = "unknown_entry";
        }
        return name + ImportLoader.EI_CONFLICT + AtpImportStrategy.IMPORT_TIMESTAMP.get();
    }

    /**
     * To quickly map between UUID and ProjectDirector, maps directory id -> ProjectDirectory.
     * Map contains sourceId and id at the same time in case if they differ,
     * Sometimes conflicts can occur between sourceId and id, e.g.
     * Given:
     * Directory1 id=1 sourceId=2
     * Directory2 id=2 sourceId=2
     * Then we have map with:
     * 1 -> Directory1
     * 2 -> Directory2
     *
     * @param projectConfiguration - ProjectConfiguration.
     * @return Map.
     */
    public static Map<UUID, ProjectDirectory> createMapIdToExistingDirectory(
            ProjectConfiguration projectConfiguration
    ) {
        return projectConfiguration.getDirectories()
                .stream()
                .flatMap(pd -> Stream.of(
                        new AbstractMap.SimpleEntry<>(pd.getSourceId(), pd),
                        new AbstractMap.SimpleEntry<>(pd.getId(), pd)
                ))
                .filter(entry -> nonNull(entry.getKey()))
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        Map.Entry::getValue,
                        (existing, duplicate) -> duplicate
                ));
    }

    /**
     * Map to quickly handle conflict with names in same folder.
     * folder id   -> Map(folder name -> ProjectDirectory)
     * UUID        -> Map(String      -> ProjectDirectory)
     * e.g. Given:
     * need to create directory2 inside directory1,
     * need to check whether directory1 already contains a subdirectory named directory2.
     * Using the UUID of directory1, access a Map (generated by this method)
     * that returns {@code Map<String, ProjectDirectory>}
     * where the keys represent the folder names within directory1.
     * If the key for directory2 exists, a duplicate is detected.
     *
     * @param projectConfiguration ProjectConfiguration.
     * @return Map of maps.
     */
    public static Map<UUID, Map<String, ProjectDirectory>> createMapParentIdToDirectoryNames(
            ProjectConfiguration projectConfiguration) {
        Map<UUID, Map<String, ProjectDirectory>> result = new HashMap<>();
        for (ProjectDirectory directory : projectConfiguration.getDirectories()) {
            result.computeIfAbsent(
                    getFolderId(directory.getParentDirectory()),
                    key -> new HashMap<>()
            ).put(directory.getName(), directory);
        }

        return result;
    }

    /**
     * Map to quickly handle conflict with names in same folder.
     * folder id   -> Map(name   -> ProjectFile)
     * UUID        -> Map(String -> ProjectFile)
     * e.g. Given:
     * need to create file.txt inside directory1,
     * need to check whether directory1 already contains a file named file.txt.
     * Using the UUID of directory1, access a Map (generated by this method)
     * that returns {@code Map<String, ProjectFile>}
     * where the keys represent the folder names within directory1.
     * If the key for file.txt exists, a duplicate is detected.
     *
     * @param projectConfiguration ProjectConfiguration.
     * @return Map of maps.
     */
    public static Map<UUID, Map<String, ProjectFile>> createMapParentIdToFileNames(
            ProjectConfiguration projectConfiguration) {
        Map<UUID, Map<String, ProjectFile>> result = new HashMap<>();
        for (ProjectFile file : projectConfiguration.getFiles()) {
            result.computeIfAbsent(
                    getFolderId(file.getDirectory()),
                    key -> new HashMap<>()
            ).put(file.getName(), file);
        }

        return result;
    }

    /**
     * To quickly map between UUID and ProjectFile, maps directory id -> ProjectFile.
     * Map contains sourceId and id at the same time in case if they differ,
     * Sometimes conflicts can occur between sourceId and id, e.g.
     * Given:
     * Directory1 id=1 sourceId=2
     * Directory2 id=2 sourceId=2
     * Then we have map with:
     * 1 -> Directory1
     * 2 -> Directory2
     *
     * @param projectConfiguration ProjectConfiguration.
     * @return Map.
     */
    public static Map<UUID, ProjectFile> createMapIdToExistingFile(ProjectConfiguration projectConfiguration) {
        return projectConfiguration.getFiles()
                .stream()
                .flatMap(pd -> Stream.of(
                        new AbstractMap.SimpleEntry<>(pd.getSourceId(), pd),
                        new AbstractMap.SimpleEntry<>(pd.getId(), pd)
                ))
                .filter(entry -> nonNull(entry.getKey()))
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        Map.Entry::getValue,
                        (existing, duplicate) -> duplicate
                ));
    }

    /**
     * Reverses key and values in map, key becomes value and value becomes key.
     *
     * @param originalMap map.
     * @return reversed map.
     */
    public static <K, V> Map<V, K> reverseMap(Map<K, V> originalMap) {
        Map<V, K> reversedMap = new HashMap<>();
        for (Map.Entry<K, V> entry : originalMap.entrySet()) {
            reversedMap.put(entry.getValue(), entry.getKey());
        }
        return reversedMap;
    }
}
