/*
 * Decompiled with CFR 0.152.
 */
package org.qubership.atp.ram.services;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.qubership.atp.ram.clients.api.dto.environments.environment.SystemFullVer1ViewDto;
import org.qubership.atp.ram.dto.request.JointExecutionRequestSearchRequest;
import org.qubership.atp.ram.dto.response.Environment;
import org.qubership.atp.ram.dto.response.JointExecutionRequestSearchResponse;
import org.qubership.atp.ram.enums.ExecutionStatuses;
import org.qubership.atp.ram.enums.TestingStatuses;
import org.qubership.atp.ram.exceptions.executionrequests.RamMultipleActiveJointExecutionRequestsException;
import org.qubership.atp.ram.models.Comment;
import org.qubership.atp.ram.models.EnvironmentsInfo;
import org.qubership.atp.ram.models.ExecutionRequest;
import org.qubership.atp.ram.models.JointExecutionRequest;
import org.qubership.atp.ram.models.LogRecord;
import org.qubership.atp.ram.models.RootCause;
import org.qubership.atp.ram.models.TestRun;
import org.qubership.atp.ram.repositories.JointExecutionRequestRepository;
import org.qubership.atp.ram.services.EnvironmentsInfoService;
import org.qubership.atp.ram.services.EnvironmentsService;
import org.qubership.atp.ram.services.ExecutionRequestService;
import org.qubership.atp.ram.services.JointExecutionRequestsReportDataModel;
import org.qubership.atp.ram.services.RootCauseService;
import org.qubership.atp.ram.services.TestRunService;
import org.qubership.atp.ram.utils.RateCalculator;
import org.qubership.atp.ram.utils.StreamUtils;
import org.qubership.atp.ram.utils.TimeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

@Service
public class JointExecutionRequestService {
    private static final Logger log = LoggerFactory.getLogger(JointExecutionRequestService.class);
    private static final String DEFAULT_PROJECT_TIME_ZONE = "GMT+03:00";
    private static final String DEFAULT_PROJECT_DATE_FORMAT = "d MMM yyyy";
    private static final String DEFAULT_PROJECT_TIME_FORMAT = "hh:mm";
    @Value(value="${catalogue.url}")
    private String catalogueUrl;
    @Value(value="${jointExecutionRequests.complete.timeout.seconds}")
    private Integer defaultTimeout;
    private final ExecutionRequestService executionRequestService;
    private final TestRunService testRunService;
    private final EnvironmentsInfoService environmentsInfoService;
    private final EnvironmentsService environmentService;
    private final RateCalculator rateCalculator;
    private final JointExecutionRequestRepository repository;
    private final RootCauseService rootCauseService;

    public void updateActiveJointExecutionRequest(ExecutionRequest executionRequest) {
        UUID executionRequestId = executionRequest.getUuid();
        log.info("Updating active joint execution request for ER with id '{}'", (Object)executionRequestId);
        JointExecutionRequest activeJointExecutionRequest = this.getActiveJointExecutionRequest(executionRequest);
        activeJointExecutionRequest.setKey(executionRequest.getJointExecutionKey());
        activeJointExecutionRequest.setCount(executionRequest.getJointExecutionCount());
        activeJointExecutionRequest.setTimeout(executionRequest.getJointExecutionTimeout());
        activeJointExecutionRequest.upsertRun(executionRequestId, executionRequest.getExecutionStatus());
        log.debug("Saving updated joint execution request");
        this.repository.save(activeJointExecutionRequest);
    }

    public JointExecutionRequest createJointExecutionRequest(ExecutionRequest executionRequest) {
        UUID executionRequestId = executionRequest.getUuid();
        log.info("Creating new joint execution request for ER with id '{}'", (Object)executionRequestId);
        JointExecutionRequest request = new JointExecutionRequest();
        Timestamp currentDate = Timestamp.valueOf(LocalDateTime.now());
        request.setName("Joint Execution Request [" + currentDate + "]");
        request.setStartDate(currentDate);
        request.setKey(executionRequest.getJointExecutionKey());
        request.setCount(executionRequest.getJointExecutionCount());
        request.setTimeout(executionRequest.getJointExecutionTimeout());
        request.setStatus(JointExecutionRequest.Status.IN_PROGRESS);
        request.upsertRun(executionRequestId, executionRequest.getExecutionStatus());
        log.debug("Saving new joint execution request: {}", (Object)request);
        return (JointExecutionRequest)this.repository.save(request);
    }

    public boolean isJointExecutionRequest(UUID executionRequestId) {
        ExecutionRequest executionRequest = (ExecutionRequest)this.executionRequestService.get(executionRequestId);
        return this.isJointExecutionRequest(executionRequest);
    }

    public boolean isJointExecutionRequest(ExecutionRequest executionRequest) {
        String jointExecutionKey = executionRequest.getJointExecutionKey();
        log.info("Checked joint execution key for ER id: {} with ststus is {}", (Object)executionRequest.getUuid(), (Object)(!StringUtils.isEmpty((Object)jointExecutionKey) ? "joint ER" : "not joint"));
        return !StringUtils.isEmpty((Object)jointExecutionKey);
    }

    public boolean isJointExecutionRequestReady(JointExecutionRequest jointExecutionRequest) {
        UUID id = jointExecutionRequest.getUuid();
        log.info("Checking is joint execution request '{}' is ready for report sending", (Object)id);
        Integer count = jointExecutionRequest.getCount();
        log.debug("Joint execution request count: {} for joint ER with id {}", (Object)count, (Object)id);
        if (count == 0) {
            log.debug("Found count is zero for joint ER with id {}", (Object)id);
            return false;
        }
        log.debug("Joint execution request runs: {}  for joint ER with id {}", (Object)jointExecutionRequest.getRuns(), (Object)id);
        List completedExecutionRequestIds = jointExecutionRequest.getCompletedExecutionRequestIds();
        long completedErs = completedExecutionRequestIds.size();
        log.debug("Completed ER's count: {}, with ids: {} for joint ER with id {}", new Object[]{completedErs, completedExecutionRequestIds, id});
        return completedErs == (long)count.intValue();
    }

    public JointExecutionRequest getActiveJointExecutionRequest(UUID executionRequestId) {
        ExecutionRequest executionRequest = (ExecutionRequest)this.executionRequestService.get(executionRequestId);
        return this.getActiveJointExecutionRequest(executionRequest);
    }

    public JointExecutionRequest getActiveJointExecutionRequest(ExecutionRequest executionRequest) {
        String jointExecutionKey = executionRequest.getJointExecutionKey();
        log.info("Getting active joint execution request for ER with id: {} and joint execution key :{}", (Object)executionRequest.getUuid(), (Object)jointExecutionKey);
        List<JointExecutionRequest> jointExecutionRequests = this.repository.findAllActiveJointExecutionRequestsByKey(jointExecutionKey);
        log.debug("Founded joint execution requests: {}", jointExecutionRequests);
        if (CollectionUtils.isEmpty(jointExecutionRequests)) {
            return this.createJointExecutionRequest(executionRequest);
        }
        if (jointExecutionRequests.size() > 1) {
            log.error("More than one active joint execution requests with the same key: {}", (Object)jointExecutionKey);
            throw new RamMultipleActiveJointExecutionRequestsException();
        }
        JointExecutionRequest jointExecutionRequest = jointExecutionRequests.get(0);
        jointExecutionRequest.upsertRun(executionRequest);
        return (JointExecutionRequest)this.repository.save(jointExecutionRequest);
    }

    public void checkAndCompleteJointExecutionRequestsByTimeout(Consumer<JointExecutionRequest> mailSendFunc) {
        log.info("Start joint execution request completion process");
        List<JointExecutionRequest> allActiveJointExecutionRequests = this.repository.findAllActiveJointExecutionRequests();
        Set<UUID> ids = StreamUtils.extractIds(allActiveJointExecutionRequests);
        log.debug("Found active joint execution requests: {}", ids);
        LocalDateTime now = LocalDateTime.now();
        log.debug("Current date: {}", (Object)now);
        allActiveJointExecutionRequests.forEach(jointExecutionRequest -> {
            log.debug("Checking joint execution request '{}'", (Object)jointExecutionRequest.getUuid());
            try {
                LocalDateTime startDate = jointExecutionRequest.getStartDate().toLocalDateTime();
                Integer timeout = jointExecutionRequest.getTimeout();
                if (timeout == 0) {
                    log.debug("Found zero timeout value for joint execution request with id: {}", (Object)jointExecutionRequest.getUuid());
                    timeout = this.defaultTimeout;
                }
                LocalDateTime timeoutDate = startDate.plusSeconds(timeout.intValue());
                log.debug("Timeout date: {}", (Object)timeoutDate);
                if (now.isAfter(timeoutDate)) {
                    log.debug("Timeout is exceeded, sending report...");
                    mailSendFunc.accept((JointExecutionRequest)jointExecutionRequest);
                    log.debug("Updating joint execution request with id {} and status {}", (Object)jointExecutionRequest.getUuid(), (Object)JointExecutionRequest.Status.COMPLETED_BY_TIMEOUT);
                    jointExecutionRequest.setStatus(JointExecutionRequest.Status.COMPLETED_BY_TIMEOUT);
                    this.repository.save(jointExecutionRequest);
                }
            }
            catch (Exception e) {
                log.error("Failed to complete by timeout joint execution request", (Throwable)e);
                this.completeFailedJointExecutionRequest((JointExecutionRequest)jointExecutionRequest, e);
            }
        });
        log.info("End joint execution request completion process");
    }

    public void updateJointExecutionRequestRunStatus(ExecutionRequest executionRequest) {
        UUID executionRequestId = executionRequest.getUuid();
        ExecutionStatuses executionStatus = executionRequest.getExecutionStatus();
        log.info("Update all active joint execution request runs for ER '{}' with status '{}'", (Object)executionRequestId, (Object)executionStatus);
        String jointExecutionKey = executionRequest.getJointExecutionKey();
        List<JointExecutionRequest> jointExecutionRequests = this.repository.findAllActiveJointExecutionRequestsByKey(jointExecutionKey);
        jointExecutionRequests.forEach(jointExecutionRequest -> jointExecutionRequest.upsertRun(executionRequestId, executionStatus));
        this.repository.saveAll(jointExecutionRequests);
    }

    public void completeJointExecutionRequest(JointExecutionRequest activeJointExecutionRequest) {
        log.info("Complete joint execution request with id '{}' and status {} ", (Object)activeJointExecutionRequest.getUuid(), (Object)JointExecutionRequest.Status.COMPLETED);
        activeJointExecutionRequest.setStatus(JointExecutionRequest.Status.COMPLETED);
        this.repository.save(activeJointExecutionRequest);
        log.debug("Joint execution request has been completed");
    }

    public void completeFailedJointExecutionRequest(JointExecutionRequest jointExecutionRequest, String ... logs) {
        log.info("Complete joint execution request with id '{}' and logs as failed", (Object)jointExecutionRequest.getUuid());
        String joinedLogs = String.join((CharSequence)"\n", logs);
        jointExecutionRequest.setLogs(joinedLogs);
        jointExecutionRequest.setStatus(JointExecutionRequest.Status.FAILED);
        log.debug("Logs: {}", (Object)joinedLogs);
        this.repository.save(jointExecutionRequest);
    }

    public void completeFailedJointExecutionRequest(JointExecutionRequest jointExecutionRequest, Exception e) {
        log.info("Complete joint execution request with id '{}' and error as failed", (Object)jointExecutionRequest.getUuid());
        String stackTrace = ExceptionUtils.getStackTrace((Throwable)e);
        jointExecutionRequest.setLogs(stackTrace);
        jointExecutionRequest.setStatus(JointExecutionRequest.Status.FAILED);
        log.debug("Stacktrace: {}", (Object)stackTrace);
        this.repository.save(jointExecutionRequest);
    }

    public JointExecutionRequestsReportDataModel getJointExecutionRequestsReportDataModel(List<UUID> erIds) {
        log.info("Getting joint execution request report data model");
        List<ExecutionRequest> executionRequests = this.executionRequestService.getOrderedExecutionRequestsByIds(erIds);
        JointExecutionRequestsReportDataModel.EnvironmentsData environmentsData = this.getEnvironmentsData(executionRequests);
        JointExecutionRequestsReportDataModel.ExecutionRequestsData executionRequestsData = this.getExecutionRequestsData(executionRequests);
        return new JointExecutionRequestsReportDataModel(environmentsData, executionRequestsData);
    }

    public JointExecutionRequest getJointExecutionRequest(String key) {
        log.info("Getting joint execution request for key: {}", (Object)key);
        List<JointExecutionRequest> jointExecutionRequests = this.repository.findAllByKey(key);
        JointExecutionRequest jointExecutionRequest = StreamUtils.checkAndReturnSingular(jointExecutionRequests);
        log.debug("Found joint execution request: {}", (Object)jointExecutionRequest);
        return jointExecutionRequest;
    }

    public List<JointExecutionRequestSearchResponse> search(JointExecutionRequestSearchRequest request) {
        log.info("Search execution requests by request: {}", (Object)request);
        String key = request.getKey();
        JointExecutionRequest jointExecutionRequest = this.getJointExecutionRequest(key);
        List completedExecutionRequestIds = jointExecutionRequest.getCompletedExecutionRequestIds();
        List<ExecutionRequest> executionRequests = this.executionRequestService.getExecutionRequestsByIds(completedExecutionRequestIds);
        log.debug("Found execution requests: {}", StreamUtils.extractIds(executionRequests));
        if (!CollectionUtils.isEmpty(executionRequests)) {
            return StreamUtils.map(executionRequests, JointExecutionRequestSearchResponse::new);
        }
        return Collections.emptyList();
    }

    private JointExecutionRequestsReportDataModel.ExecutionRequestsData getExecutionRequestsData(List<ExecutionRequest> executionRequests) {
        JointExecutionRequestsReportDataModel.ExecutionRequestsData executionRequestsData = new JointExecutionRequestsReportDataModel.ExecutionRequestsData();
        List<JointExecutionRequestsReportDataModel.ExecutionRequestCount> executionRequestCounts = this.getExecutionRequestCounts(executionRequests);
        executionRequestsData.setExecutionRequestCounts(executionRequestCounts);
        int totalTestCaseCount = executionRequestCounts.stream().mapToInt(JointExecutionRequestsReportDataModel.ExecutionRequestCount::getTestCaseCount).sum();
        Map<String, Long> statusTotalCount = executionRequestCounts.stream().flatMap(executionRequestCount -> executionRequestCount.getStatusCounts().stream()).collect(Collectors.groupingBy(JointExecutionRequestsReportDataModel.StatusCount::getStatus, Collectors.summingLong(JointExecutionRequestsReportDataModel.StatusCount::getCount)));
        List<JointExecutionRequestsReportDataModel.StatusCount> statusOverallCounts = statusTotalCount.entrySet().stream().sorted(Comparator.comparingInt(entry -> TestingStatuses.findByName((String)((String)entry.getKey())).getReportsOrder())).map(entry -> {
            String status = (String)entry.getKey();
            int totalStatusCount = Math.toIntExact((Long)entry.getValue());
            float rate = RateCalculator.calculateRateFloat(totalStatusCount, totalTestCaseCount);
            return new JointExecutionRequestsReportDataModel.StatusCount(status, totalStatusCount, rate);
        }).collect(Collectors.toList());
        List<String> statuses = Arrays.stream(TestingStatuses.values()).sorted(Comparator.comparingInt(TestingStatuses::getReportsOrder)).map(testingStatus -> testingStatus.getName().toUpperCase()).collect(Collectors.toList());
        executionRequestsData.setStatuses(statuses);
        executionRequestsData.setTotalStatusCounts(statusOverallCounts);
        executionRequestsData.setTestCaseTotalCount(totalTestCaseCount);
        return executionRequestsData;
    }

    private List<JointExecutionRequestsReportDataModel.ExecutionRequestCount> getExecutionRequestCounts(List<ExecutionRequest> executionRequests) {
        return executionRequests.stream().map(executionRequest -> {
            UUID executionRequestId = executionRequest.getUuid();
            UUID projectId = executionRequest.getProjectId();
            List<TestRun> testRuns = this.executionRequestService.getAllTestRuns(executionRequestId);
            JointExecutionRequestsReportDataModel.ExecutionRequestCount executionRequestCount = new JointExecutionRequestsReportDataModel.ExecutionRequestCount();
            List<JointExecutionRequestsReportDataModel.StatusCount> statusCounts = this.getStatusCounts((ExecutionRequest)executionRequest, testRuns);
            List<JointExecutionRequestsReportDataModel.TestCase> testCases = this.getTestCases(testRuns);
            executionRequestCount.setStatusCounts(statusCounts);
            executionRequestCount.setTestCaseCount(executionRequest.getCountOfTestRuns());
            executionRequestCount.setName(executionRequest.getName());
            executionRequestCount.setLink(this.getExecutionRequestLink(this.catalogueUrl, projectId, executionRequestId));
            executionRequestCount.setTestCases(testCases);
            return executionRequestCount;
        }).collect(Collectors.toList());
    }

    private List<JointExecutionRequestsReportDataModel.TestCase> getTestCases(List<TestRun> testRuns) {
        Set<UUID> rootCauseIds = StreamUtils.extractIds(testRuns, TestRun::getRootCauseId);
        List<RootCause> rootCauses = this.rootCauseService.getByIds(rootCauseIds);
        Map<UUID, String> rootCauseMap = StreamUtils.toIdNameEntityMap(rootCauses);
        return testRuns.stream().map(testRun -> {
            UUID rootCauseId;
            Comment comment;
            UUID testRunId = testRun.getUuid();
            JointExecutionRequestsReportDataModel.TestCase testCase = new JointExecutionRequestsReportDataModel.TestCase();
            testCase.setName(testRun.getName());
            TestingStatuses testingStatus = testRun.getTestingStatus();
            testCase.setStatus(testingStatus.getName().toUpperCase());
            String duration = DurationFormatUtils.formatDuration((long)(testRun.getDuration() * 1000L), (String)"HH:mm:ss", (boolean)true);
            testCase.setDuration(duration);
            if (testingStatus.equals((Object)TestingStatuses.FAILED)) {
                LogRecord firstFailedStep = this.testRunService.getFirstFailedStep(testRunId);
                testCase.setFirstFailedStepName(firstFailedStep.getName());
            }
            if (Objects.nonNull(comment = testRun.getComment())) {
                testCase.setComment(comment.getHtml());
            }
            if (Objects.nonNull(rootCauseId = testRun.getRootCauseId())) {
                String rootCauseName = (String)rootCauseMap.get(rootCauseId);
                testCase.setFailureReason(rootCauseName);
            }
            return testCase;
        }).collect(Collectors.toList());
    }

    private List<JointExecutionRequestsReportDataModel.StatusCount> getStatusCounts(ExecutionRequest executionRequest, List<TestRun> testRuns) {
        Map<TestingStatuses, RateCalculator.TestingStatusesStat> stats = this.rateCalculator.calculateTestRunsTestingStatusStats(executionRequest, testRuns);
        return stats.entrySet().stream().sorted(Comparator.comparingInt(entry -> ((TestingStatuses)entry.getKey()).getReportsOrder())).map(entry -> {
            TestingStatuses status = (TestingStatuses)entry.getKey();
            RateCalculator.TestingStatusesStat statusStats = (RateCalculator.TestingStatusesStat)entry.getValue();
            return new JointExecutionRequestsReportDataModel.StatusCount(status.getName(), statusStats.getCount(), statusStats.getRate());
        }).collect(Collectors.toList());
    }

    private JointExecutionRequestsReportDataModel.EnvironmentsData getEnvironmentsData(List<ExecutionRequest> executionRequests) {
        Map<UUID, ExecutionRequest> executionRequestMap = StreamUtils.toIdEntityMap(executionRequests);
        Set<UUID> executionRequestIds = StreamUtils.extractIds(executionRequests);
        List<EnvironmentsInfo> environmentsInfos = this.environmentsInfoService.findByRequestIds(executionRequestIds);
        JointExecutionRequestsReportDataModel.EnvironmentsData environmentsData = new JointExecutionRequestsReportDataModel.EnvironmentsData();
        List<JointExecutionRequestsReportDataModel.EnvironmentData> environments = environmentsInfos.stream().map(environmentsInfo -> {
            JointExecutionRequestsReportDataModel.EnvironmentData environmentData = new JointExecutionRequestsReportDataModel.EnvironmentData();
            UUID executionRequestId = environmentsInfo.getExecutionRequestId();
            ExecutionRequest executionRequest = (ExecutionRequest)executionRequestMap.get(executionRequestId);
            UUID projectId = executionRequest.getProjectId();
            UUID environmentId = environmentsInfo.getEnvironmentId();
            String environmentLink = this.getEnvironmentLink(this.catalogueUrl, projectId, environmentId);
            List qaSystems = environmentsInfo.getQaSystemInfoList();
            List<SystemFullVer1ViewDto> systems = this.environmentService.getEnvironmentSystems(environmentId);
            Map systemsMap = systems.stream().collect(Collectors.toMap(SystemFullVer1ViewDto::getName, Function.identity()));
            List<JointExecutionRequestsReportDataModel.QaSystemData> qaSystemsData = qaSystems.stream().map(qaSystem -> {
                String name = qaSystem.getName();
                String version = qaSystem.getVersion();
                SystemFullVer1ViewDto system = (SystemFullVer1ViewDto)systemsMap.get(name);
                String timestamp = this.getLastVersionCheckDate(system);
                return new JointExecutionRequestsReportDataModel.QaSystemData(name, version, timestamp);
            }).collect(Collectors.toList());
            Environment environment = this.environmentService.getEnvironmentById(environmentId);
            environmentData.setName(environment.getName());
            environmentData.setLink(environmentLink);
            environmentData.setSystems(qaSystemsData);
            return environmentData;
        }).collect(Collectors.toList());
        environmentsData.setEnvironments(environments);
        return environmentsData;
    }

    private String getLastVersionCheckDate(SystemFullVer1ViewDto system) {
        Long lastVersionCheck = system.getDateOfLastCheck();
        String timestamp = null;
        if (lastVersionCheck != null) {
            String dateTimeFormat = String.format("%s %s", DEFAULT_PROJECT_DATE_FORMAT, DEFAULT_PROJECT_TIME_FORMAT);
            Timestamp versionTimestamp = new Timestamp(lastVersionCheck);
            timestamp = TimeUtils.formatDateTime(versionTimestamp, dateTimeFormat, DEFAULT_PROJECT_TIME_ZONE);
        }
        return timestamp;
    }

    private String getExecutionRequestLink(String host, UUID projectId, UUID executionRequestId) {
        return host + "/project/" + projectId + "/ram/execution-request/" + executionRequestId;
    }

    private String getEnvironmentLink(String host, UUID projectId, UUID environmentId) {
        return host + "/project/" + projectId + "/environments/environment/" + environmentId;
    }

    public JointExecutionRequestService(ExecutionRequestService executionRequestService, TestRunService testRunService, EnvironmentsInfoService environmentsInfoService, EnvironmentsService environmentService, RateCalculator rateCalculator, JointExecutionRequestRepository repository, RootCauseService rootCauseService) {
        this.executionRequestService = executionRequestService;
        this.testRunService = testRunService;
        this.environmentsInfoService = environmentsInfoService;
        this.environmentService = environmentService;
        this.rateCalculator = rateCalculator;
        this.repository = repository;
        this.rootCauseService = rootCauseService;
    }
}

