/*
 * Decompiled with CFR 0.152.
 */
package org.qubership.atp.ram.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.validation.Valid;
import org.modelmapper.ModelMapper;
import org.qubership.atp.auth.springbootstarter.exceptions.AtpException;
import org.qubership.atp.auth.springbootstarter.utils.ExceptionUtils;
import org.qubership.atp.ram.exceptions.rootcauses.RamRootCauseAlreadyExistsException;
import org.qubership.atp.ram.exceptions.rootcauses.RamRootCauseIllegalAccessException;
import org.qubership.atp.ram.exceptions.rootcauses.RamRootCauseIllegalTypeException;
import org.qubership.atp.ram.model.request.RootCauseUpsertRequest;
import org.qubership.atp.ram.models.RootCause;
import org.qubership.atp.ram.models.RootCauseTreeNode;
import org.qubership.atp.ram.models.RootCauseType;
import org.qubership.atp.ram.repositories.RootCauseRepository;
import org.qubership.atp.ram.services.CrudService;
import org.qubership.atp.ram.services.TestRunService;
import org.qubership.atp.ram.utils.SecurityUtils;
import org.qubership.atp.ram.utils.StreamUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.mongodb.repository.MongoRepository;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

@Service
public class RootCauseService
extends CrudService<RootCause> {
    private static final Logger log = LoggerFactory.getLogger(RootCauseService.class);
    private final RootCauseRepository repository;
    private final TestRunService testRunService;
    private final ModelMapper modelMapper;

    public RootCauseService(RootCauseRepository repository, @Lazy TestRunService testRunService, ModelMapper modelMapper) {
        this.repository = repository;
        this.testRunService = testRunService;
        this.modelMapper = modelMapper;
    }

    @Override
    protected MongoRepository<RootCause, UUID> repository() {
        return this.repository;
    }

    @Cacheable(value={"rootcauses"})
    public List<RootCause> getAllRootCauses() {
        log.info("Get all root causes");
        return this.repository.findAll();
    }

    @Cacheable(value={"rootcauses"})
    public RootCause getById(UUID id) {
        log.info("Get root cause by id '{}'", (Object)id);
        return (RootCause)this.get(id);
    }

    public RootCause getByIdOrNull(UUID id) {
        return this.repository().findById((Object)id).orElse(null);
    }

    @CacheEvict(value={"rootcauses"}, allEntries=true)
    public RootCause create(@Valid RootCauseUpsertRequest request) {
        log.info("Create root cause with content: {}", (Object)request);
        RootCause savedRootCause = (RootCause)this.modelMapper.map((Object)request, RootCause.class);
        this.validateRootCauseForUserRights(savedRootCause);
        this.validateRootCauseNameUniqueness(savedRootCause);
        return (RootCause)this.repository.save(savedRootCause);
    }

    @CacheEvict(value={"rootcauses"}, allEntries=true)
    public RootCause update(UUID id, @Valid RootCauseUpsertRequest request) {
        log.info("Update root cause '{}' with content: {}", (Object)id, (Object)request);
        RootCause existedRootCause = (RootCause)this.get(id);
        this.modelMapper.map((Object)request, (Object)existedRootCause);
        this.validateRootCauseForUserRights(existedRootCause);
        this.validateRootCauseNameUniqueness(existedRootCause);
        return (RootCause)this.repository.save(existedRootCause);
    }

    private void validateRootCauseForUserRights(RootCause rootCause) {
        boolean nonAdmin;
        boolean isGlobalRootCause = RootCauseType.GLOBAL.equals((Object)rootCause.getType());
        boolean bl = nonAdmin = !SecurityUtils.isCurrentUserAdmin();
        if (isGlobalRootCause && nonAdmin) {
            ExceptionUtils.throwWithLog((Logger)log, (AtpException)new RamRootCauseIllegalAccessException());
        }
    }

    private void validateRootCauseNameUniqueness(RootCause savedRootCause) {
        RootCause existedRootCause;
        RootCauseType type = savedRootCause.getType();
        String name = savedRootCause.getName();
        UUID projectId = savedRootCause.getProjectId();
        UUID id = savedRootCause.getUuid();
        switch (type) {
            case CUSTOM: {
                existedRootCause = this.repository.findByNameAndProjectId(name, projectId);
                break;
            }
            case GLOBAL: {
                existedRootCause = this.repository.findByNameAndType(name, RootCauseType.GLOBAL);
                break;
            }
            default: {
                log.error("Found illegal root cause type: {}", (Object)type);
                throw new RamRootCauseIllegalTypeException(type);
            }
        }
        if (Objects.nonNull(existedRootCause)) {
            boolean isNotUpdatedRootCause;
            boolean isSameNames = name.equals(existedRootCause.getName());
            boolean bl = isNotUpdatedRootCause = Objects.nonNull(id) && !id.equals(existedRootCause.getUuid());
            if (isSameNames && (Objects.isNull(id) || isNotUpdatedRootCause)) {
                log.error("Global root cause with provided name '{}' already exists", (Object)name);
                throw new RamRootCauseAlreadyExistsException();
            }
        }
    }

    @CacheEvict(value={"rootcauses"}, allEntries=true)
    public void deleteById(UUID id) {
        log.info("Delete root cause by id '{}'", (Object)id);
        RootCause rootCause = (RootCause)this.get(id);
        this.deleteRootCauseHierarchy(rootCause);
    }

    public void deleteRootCauseHierarchy(RootCause rootCause) {
        UUID rootCauseId = rootCause.getUuid();
        log.info("Start hierarchical deletion for parent root cause with id '{}'", (Object)rootCauseId);
        List<RootCause> children = this.getChildrenRootCauses(rootCause, null);
        if (!CollectionUtils.isEmpty(children)) {
            children.forEach(this::deleteRootCauseHierarchy);
        }
        log.debug("Delete root cause with id '{}'", (Object)rootCauseId);
        this.testRunService.unsetRootCauseForLinkedTestRuns(rootCauseId);
        this.repository.delete(rootCause);
    }

    @Cacheable(value={"rootcauses"})
    public List<RootCause> getByIds(Collection<UUID> ids) {
        log.info("Get root causes by ids [{}]", ids);
        return this.repository.findByUuidIn(ids);
    }

    @Cacheable(value={"rootcauses"})
    public List<RootCauseTreeNode> getRootCauseTree(UUID projectId, boolean filterDisabled) {
        log.info("Get root cause tree for project '{}'", (Object)projectId);
        List<RootCause> globalRootCauses = this.getAllGlobalTopLevelRootCauses();
        List<RootCause> projectCustomRootCauses = this.getAllTopLevelCustomProjectRootCauses(projectId);
        ArrayList<RootCause> allTopLevelRootCauses = new ArrayList<RootCause>();
        allTopLevelRootCauses.addAll(globalRootCauses);
        allTopLevelRootCauses.addAll(projectCustomRootCauses);
        return this.getTreeByTopLevelRootCauses(allTopLevelRootCauses, projectId, filterDisabled);
    }

    public String getRootCauseNameById(UUID id) {
        if (Objects.nonNull(id)) {
            return this.repository.findNameByUuid(id).getName();
        }
        return "";
    }

    private List<RootCauseTreeNode> getTreeByTopLevelRootCauses(List<RootCause> topLevelRootCauses, UUID projectId, boolean filterDisabled) {
        return topLevelRootCauses.stream().filter(rootCause -> !filterDisabled || !rootCause.isDisabled()).map(rootCause -> this.getTreeNode((RootCause)rootCause, projectId, filterDisabled)).collect(Collectors.toList());
    }

    private RootCauseTreeNode getTreeNode(RootCause rootCause, UUID projectId, boolean filterDisabled) {
        log.debug("Get tree node for root cause: {}", (Object)rootCause);
        RootCauseTreeNode parentNode = new RootCauseTreeNode(rootCause);
        List<RootCause> children = this.getChildrenRootCauses(rootCause, projectId);
        if (!CollectionUtils.isEmpty(children)) {
            List childrenNodes = children.stream().filter(childRootCause -> !filterDisabled || !childRootCause.isDisabled()).map(childRootCause -> this.getTreeNode((RootCause)childRootCause, projectId, filterDisabled)).collect(Collectors.toList());
            parentNode.setChildren(childrenNodes);
        }
        return parentNode;
    }

    public List<RootCause> getChildrenRootCauses(RootCause parentRootCause, UUID projectId) {
        log.debug("Get children root causes for parent root cause '{}'", (Object)parentRootCause);
        UUID parentRootCauseId = parentRootCause.getUuid();
        RootCauseType rootCauseType = parentRootCause.getType();
        switch (rootCauseType) {
            case CUSTOM: {
                return this.repository.findAllByParentIdAndProjectId(parentRootCauseId, projectId);
            }
            case GLOBAL: {
                return this.repository.findAllByParentId(parentRootCauseId);
            }
        }
        return Collections.emptyList();
    }

    public boolean isNameUsed(RootCause rootCause, UUID targetProjectId) {
        List<RootCause> fromBase = rootCause.getProjectId() == null ? this.repository.findAllByNameAndParentId(rootCause.getName(), rootCause.getParentId()) : (targetProjectId.equals(rootCause.getProjectId()) ? this.repository.findAllByProjectIdAndNameAndParentId(rootCause.getProjectId(), rootCause.getName(), rootCause.getParentId()) : this.repository.findAllByNameAndParentId(rootCause.getName(), rootCause.getParentId()));
        return !CollectionUtils.isEmpty(fromBase) && fromBase.stream().noneMatch(base -> base.getUuid().equals(rootCause.getUuid()));
    }

    private List<RootCause> getAllTopLevelCustomProjectRootCauses(UUID projectId) {
        log.debug("Getting all project custom top level root causes");
        List<RootCause> rootCauses = this.repository.findAllByParentIdIsNullAndProjectIdAndType(projectId, RootCauseType.CUSTOM);
        log.debug("Found root causes: [{}]", StreamUtils.extractIds(rootCauses));
        return rootCauses;
    }

    private List<RootCause> getAllGlobalTopLevelRootCauses() {
        log.debug("Getting all global root causes");
        List<RootCause> rootCauses = this.repository.findAllByParentIdIsNullAndType(RootCauseType.GLOBAL);
        log.debug("Found root causes: [{}]", StreamUtils.extractIds(rootCauses));
        return rootCauses;
    }

    @CacheEvict(value={"rootcauses"}, allEntries=true)
    public void disable(UUID id) {
        this.updateDisableStatus(id, true);
        log.debug("Root cause with id '{}' has been disabled", (Object)id);
    }

    @CacheEvict(value={"rootcauses"}, allEntries=true)
    public void enable(UUID id) {
        log.info("Enable root cause with id '{}'", (Object)id);
        this.updateDisableStatus(id, false);
        log.debug("Root cause with id '{}' has been disabled", (Object)id);
    }

    private void updateDisableStatus(UUID id, boolean status) {
        RootCause rootCause = (RootCause)this.get(id);
        rootCause.setDisabled(status);
        this.repository.save(rootCause);
    }

    public List<RootCause> getRootCausesByParentId(UUID parentId) {
        if (parentId == null) {
            return Collections.emptyList();
        }
        log.debug("Getting root causes by parentId {}", (Object)parentId);
        List<RootCause> foundRootCauses = this.repository.findAllByParentId(parentId);
        log.debug("Found root causes: [{}]", StreamUtils.extractIds(foundRootCauses));
        return foundRootCauses;
    }

    public boolean existsById(UUID id) {
        return this.repository.existsById(id);
    }
}

