package org.qubership.atp.ram.utils;

import java.lang.reflect.Field;

import javax.annotation.PostConstruct;

import org.qubership.atp.ram.clients.api.dto.catalogue.BugTrackingSystemSynchronizationDto;
import org.qubership.atp.ram.clients.api.dto.catalogue.BugTrackingSystemSynchronizationDtoDto;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import lombok.extern.slf4j.Slf4j;

@Component
@Slf4j
public class SystemTypeEnumReplacer {

    @Value("${internal.system-type.alias}")
    private String alias;

    /**
     * Patches the string “wire values” of OpenAPI-generated inner enums (SystemTypeEnum)
     * to ensure compatibility with external services.
     * <p>
     * Background: OpenAPI enums in these DTOs keep a private, non-final {@code value} field
     * that is used by {@code @JsonCreator fromValue(...)} and {@code @JsonValue getValue()}.
     * This method replaces that field via reflection with the alias from the
     * {@code internal.system-type.alias} property (e.g., {@code INTERNAL_JIRA}).
     * After the patch, Jackson will both accept and emit the alias.
     * </p>
     */
    @PostConstruct
    public void patch() {
        patchEnumValue(
                BugTrackingSystemSynchronizationDto.SystemTypeEnum.class,
                BugTrackingSystemSynchronizationDto.SystemTypeEnum.INTERNAL_JIRA
        );
        log.debug("BugTrackingSystemSynchronizationDto.SystemTypeEnum.INTERNAL_JIRA.getValue() = {}",
                BugTrackingSystemSynchronizationDto.SystemTypeEnum.INTERNAL_JIRA.getValue());

        patchEnumValue(
                BugTrackingSystemSynchronizationDtoDto.SystemTypeEnum.class,
                BugTrackingSystemSynchronizationDtoDto.SystemTypeEnum.INTERNAL_JIRA
        );
        log.debug("BugTrackingSystemSynchronizationDtoDto.SystemTypeEnum.INTERNAL_JIRA.getValue() = {}",
                BugTrackingSystemSynchronizationDtoDto.SystemTypeEnum.INTERNAL_JIRA.getValue());
    }

    /**
     * Replaces the private {@code value} field of the given enum constant with the configured alias.
     * This effectively rebinds the enum’s external (de)serialization string for Jackson.
     *
     * @param enumClass                 the inner enum class generated by OpenAPI
     * @param systemTypeEnum            the specific enum constant whose {@code value} should be replaced
     * @throws IllegalStateException    if the {@code value} field is missing, inaccessible,
     *                                  or cannot be updated
     */
    private void patchEnumValue(Class<?> enumClass, Enum<?> systemTypeEnum) {
        try {
            Field field = enumClass.getDeclaredField("value");
            field.setAccessible(true);
            field.set(systemTypeEnum, alias);
        } catch (Exception e) {
            throw new IllegalStateException("Failed to patch " + enumClass.getName() + "#" + systemTypeEnum, e);
        }
    }
}
