/*
 * Decompiled with CFR 0.152.
 */
package org.qubership.atp.ei.node.services;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.qubership.atp.ei.node.config.ExportImportNodeConfig;
import org.qubership.atp.ei.node.exceptions.ExportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.InputStreamSource;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.FileSystemUtils;
import org.springframework.util.StreamUtils;

@Service
public class FileService {
    private static final Logger log = LoggerFactory.getLogger(FileService.class);
    private static final String ATTRIBUTE_LAST_MOD_NAME = "lastModifiedTime";

    public void deletePath(Path dir) throws ExportException {
        if (dir == null) {
            return;
        }
        log.debug("delete path {}", (Object)dir);
        try {
            FileSystemUtils.deleteRecursively((Path)dir);
        }
        catch (IOException e) {
            log.error("Cannot delete dir {} ", (Object)dir, (Object)e);
        }
    }

    public void deleteFolder(UUID projectId, String taskId, String folderName) {
        Path folder = this.getFolderPath(projectId, taskId, folderName);
        this.deletePath(folder);
    }

    public void copyPath(Path dir, Path dest) throws ExportException {
        Assert.notNull((Object)dir, (String)"Argument dir is null");
        Assert.notNull((Object)dest, (String)"Argument dest is null");
        log.debug("copy path {} to {}", (Object)dir.toString(), (Object)dest.toString());
        try {
            FileSystemUtils.copyRecursively((Path)dir, (Path)dest);
        }
        catch (IOException e) {
            log.error("Cannot copy dir {} to {}", new Object[]{dir, dest, e});
            ExportException.throwException("Cannot copy dir {} to ", dir, dest, e);
        }
    }

    public Path createDirectory(String name, Path parent) throws ExportException {
        Assert.notNull((Object)parent, (String)"Argument parent is null");
        Path result = parent.resolve(name);
        return this.createDirectory(result);
    }

    public Path createDirectory(Path dir) throws ExportException {
        Assert.notNull((Object)dir, (String)"Argument dir is null");
        log.debug("create dir {}", (Object)dir.toString());
        try {
            if (!Files.exists(dir, new LinkOption[0])) {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            log.error("Cannot create dir {} ", (Object)dir.toString(), (Object)e);
            ExportException.throwException("Cannot create dir {} ", dir.toString(), e);
        }
        return dir;
    }

    public Path createFile(String name, Path parent) throws ExportException {
        Assert.notNull((Object)parent, (String)"Argument parent is null");
        Path file = parent.resolve(name);
        return this.createFile(file);
    }

    public Path createFile(Path file) throws ExportException {
        Assert.notNull((Object)file, (String)"Argument file is null");
        log.debug("create file {}", (Object)file.toString());
        try {
            if (!Files.exists(file, new LinkOption[0])) {
                Files.createFile(file, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            log.error("Cannot create dir {} ", (Object)file.toString(), (Object)e);
            ExportException.throwException("Cannot create dir {} ", file.toString(), e);
        }
        return file;
    }

    public Path packDirectory(final Path dirToZip) throws ExportException {
        Assert.notNull((Object)dirToZip, (String)"Argument dirToZip is null");
        log.debug("pack dir {}", (Object)dirToZip.toString());
        Path parentDir = dirToZip.getParent();
        String archiveFileName = dirToZip.getFileName().toString();
        Path archiveFile = Paths.get(parentDir.toAbsolutePath().toString(), archiveFileName + ".zip");
        this.deletePath(archiveFile);
        try (OutputStream fos = Files.newOutputStream(archiveFile, new OpenOption[0]);
             final ZipOutputStream zipOut = new ZipOutputStream(fos);){
            Files.walkFileTree(dirToZip, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (!dirToZip.equals(dir)) {
                        zipOut.putNextEntry(new ZipEntry(dirToZip.relativize(dir).toString() + "/"));
                        zipOut.closeEntry();
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    zipOut.putNextEntry(new ZipEntry(dirToZip.relativize(file).toString()));
                    Files.copy(file, zipOut);
                    zipOut.closeEntry();
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            log.error("Cannot zip dir {}", (Object)dirToZip, (Object)e);
            ExportException.throwException("Cannot zip dir {}", dirToZip, e);
        }
        return archiveFile;
    }

    public boolean isDirNotEmpty(Path directory) {
        boolean isNotEmpty = false;
        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);){
            isNotEmpty = dirStream.iterator().hasNext();
        }
        catch (Exception e) {
            log.error("Cannot read directory {}", (Object)directory, (Object)e);
        }
        return isNotEmpty;
    }

    public Path unpackZipFile(Path zipFile, Path dest) throws ExportException {
        Assert.notNull((Object)zipFile, (String)"Argument zipFile is null");
        Assert.notNull((Object)dest, (String)"Argument dest is null");
        log.debug("unpack zip {} in {}", (Object)zipFile.toString(), (Object)dest.toString());
        if (!Files.exists(dest, new LinkOption[0])) {
            this.createDirectory(dest);
        }
        byte[] buffer = new byte[1024];
        try (InputStream in = Files.newInputStream(zipFile, new OpenOption[0]);
             ZipInputStream zis = new ZipInputStream(in);){
            ZipEntry zipEntry = zis.getNextEntry();
            while (zipEntry != null) {
                Path newFile = dest.resolve(zipEntry.getName());
                if (zipEntry.isDirectory()) {
                    this.createDirectory(newFile);
                } else {
                    this.createDirectory(newFile.getParent());
                    this.createFile(newFile);
                    try (OutputStream fos = Files.newOutputStream(newFile, new OpenOption[0]);){
                        int len;
                        while ((len = zis.read(buffer)) > 0) {
                            fos.write(buffer, 0, len);
                        }
                    }
                }
                zipEntry = zis.getNextEntry();
            }
        }
        catch (IOException e) {
            log.error("Cannot unzip archive {} in dir {}", new Object[]{zipFile, dest, e});
            ExportException.throwException("Cannot unzip archive {} in dir {}", zipFile, dest, e);
        }
        return dest;
    }

    public Path getFolderPath(UUID projectId, String taskId, String exportOrImportFolderName) {
        return Paths.get(ExportImportNodeConfig.DEFAULT_WORK_DIR, new String[0]).resolve(projectId.toString()).resolve(exportOrImportFolderName).resolve(taskId);
    }

    public Path getFilePath(UUID projectId, String taskId, String exportOrImportFolderName) {
        Path workDir = this.getFolderPath(projectId, taskId, exportOrImportFolderName);
        return workDir.getParent().resolve(taskId + ".zip");
    }

    public Path getFilePathIfExists(UUID projectId, String taskId, String exportOrImportFolderName) {
        Path archive = this.getFilePath(projectId, taskId, exportOrImportFolderName);
        if (Files.exists(archive, new LinkOption[0])) {
            return archive;
        }
        return null;
    }

    public void saveInputStreamSourceInFile(InputStreamSource inputStreamSource, Path file) throws ExportException {
        try (InputStream in = inputStreamSource.getInputStream();
             OutputStream out = Files.newOutputStream(file, new OpenOption[0]);){
            StreamUtils.copy((InputStream)in, (OutputStream)out);
        }
        catch (IOException e) {
            log.error("Can not save input stream in file {}", (Object)file, (Object)e);
            ExportException.throwException("Can not save input stream in file {}", file, e);
        }
    }

    public List<Path> findAllArchives(Path workDir) {
        ArrayList<Path> archives = new ArrayList<Path>();
        try (Stream<Path> files = Files.find(workDir, 1, (path, basicFileAttributes) -> basicFileAttributes.isRegularFile() && path.getFileName().toString().contains(".zip"), new FileVisitOption[0]);){
            archives.addAll(files.collect(Collectors.toList()));
        }
        catch (IOException e) {
            log.error("Cannot find export files in dir {}", (Object)workDir, (Object)e);
            ExportException.throwException("Cannot find export files in dir {}", workDir, e);
        }
        return archives;
    }

    public List<String> removeAllOutdatedFilesAndFolders(Path workDir, Long expirationTimeInMilliseconds) {
        final Long expirationDate = System.currentTimeMillis() - expirationTimeInMilliseconds;
        final ArrayList<String> deletedPaths = new ArrayList<String>();
        if (Objects.isNull(workDir) || !Files.exists(workDir, new LinkOption[0])) {
            log.warn("Can not delete old files in {} directory, because it doesn't exist", (Object)workDir);
            return deletedPaths;
        }
        try {
            Files.walkFileTree(workDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (attrs.lastModifiedTime().toMillis() < expirationDate) {
                        deletedPaths.add(file.toString());
                        Files.delete(file);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc != null) {
                        log.error("An error occurred while deleting old files in folder {}", (Object)dir, (Object)exc);
                    }
                    if (!FileService.this.isDirNotEmpty(dir)) {
                        deletedPaths.add(dir.toString());
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            log.error("Can not delete files older than {} millis in {} dir", new Object[]{expirationTimeInMilliseconds, workDir, e});
            ExportException.throwException("Can not delete files older than {} millis in {} dir", expirationTimeInMilliseconds, workDir, e);
        }
        return deletedPaths;
    }
}

