/*
 * Decompiled with CFR 0.152.
 */
package org.qubership.atp.tdm.service.impl;

import com.google.common.base.Preconditions;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.quartz.CronExpression;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.qubership.atp.integration.configuration.mdc.MdcUtils;
import org.qubership.atp.tdm.env.configurator.model.Server;
import org.qubership.atp.tdm.env.configurator.service.EnvironmentsService;
import org.qubership.atp.tdm.exceptions.db.TdmDbExecuteQueryException;
import org.qubership.atp.tdm.exceptions.internal.TdmSearchImportInfoException;
import org.qubership.atp.tdm.exceptions.internal.TdmSearchRefreshConfigException;
import org.qubership.atp.tdm.mdc.MdcField;
import org.qubership.atp.tdm.mdc.TdmMdcHelper;
import org.qubership.atp.tdm.model.TestDataTableCatalog;
import org.qubership.atp.tdm.model.TestDataTableImportInfo;
import org.qubership.atp.tdm.model.refresh.RefreshResults;
import org.qubership.atp.tdm.model.refresh.TestDataRefreshConfig;
import org.qubership.atp.tdm.model.scheduler.DataRefreshJob;
import org.qubership.atp.tdm.repo.CatalogRepository;
import org.qubership.atp.tdm.repo.ImportInfoRepository;
import org.qubership.atp.tdm.repo.RefreshConfigRepository;
import org.qubership.atp.tdm.repo.SqlRepository;
import org.qubership.atp.tdm.repo.TestDataTableRepository;
import org.qubership.atp.tdm.service.DataRefreshService;
import org.qubership.atp.tdm.service.SchedulerService;
import org.qubership.atp.tdm.service.impl.MetricService;
import org.qubership.atp.tdm.utils.ValidateCronExpression;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
public class DataRefreshServiceImpl
implements DataRefreshService {
    private static final Logger log = LoggerFactory.getLogger(DataRefreshServiceImpl.class);
    private static final String SCHED_GROUP = "refresh";
    @Value(value="${external.query.default.timeout:1800}")
    private Integer defaultQueryTimeout;
    @Value(value="${external.query.max.timeout:3600}")
    private Integer maxQueryTimeout;
    private final EnvironmentsService environmentsService;
    private final SchedulerService schedulerService;
    private final RefreshConfigRepository refreshConfigRepository;
    private final TestDataTableRepository testDataTableRepository;
    private final ImportInfoRepository importInfoRepository;
    private final CatalogRepository catalogRepository;
    private final SqlRepository sqlRepository;
    private final MetricService metricService;
    private final TdmMdcHelper tdmMdcHelper;

    @Autowired
    public DataRefreshServiceImpl(@Nonnull EnvironmentsService environmentsService, @Nonnull SchedulerService schedulerService, @Nonnull RefreshConfigRepository repository, @Nonnull TestDataTableRepository testDataTableRepository, @Nonnull ImportInfoRepository importInfoRepository, @Nonnull CatalogRepository catalogRepository, @Nonnull SqlRepository sqlRepository, @Nonnull MetricService metricService, TdmMdcHelper helper) {
        this.environmentsService = environmentsService;
        this.schedulerService = schedulerService;
        this.refreshConfigRepository = repository;
        this.testDataTableRepository = testDataTableRepository;
        this.importInfoRepository = importInfoRepository;
        this.catalogRepository = catalogRepository;
        this.sqlRepository = sqlRepository;
        this.metricService = metricService;
        this.tdmMdcHelper = helper;
    }

    @Override
    public TestDataRefreshConfig getRefreshConfig(@Nonnull UUID id) {
        TestDataRefreshConfig testDataRefreshConfig = (TestDataRefreshConfig)this.refreshConfigRepository.findById(id).orElseThrow(() -> new TdmSearchRefreshConfigException(id.toString()));
        testDataRefreshConfig.setQueryTimout(this.getQueryTimeout(id));
        return testDataRefreshConfig;
    }

    private Integer getQueryTimeout(UUID id) {
        TestDataTableCatalog tableCatalog = this.catalogRepository.findByRefreshConfigId(id);
        TestDataTableImportInfo importInfo = this.importInfoRepository.findByTableName(tableCatalog.getTableName());
        return (Integer)ObjectUtils.defaultIfNull((Object)importInfo.getQueryTimeout(), (Object)this.defaultQueryTimeout);
    }

    @Override
    public TestDataRefreshConfig saveRefreshConfig(@Nonnull String tableName, @Nonnull Integer queryTimeout, @Nonnull TestDataRefreshConfig config) throws Exception {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)tableName), (Object)"Table Name is null");
        Preconditions.checkArgument((boolean)this.checkSqlAvailability(tableName), (Object)"Environment ins't set up properly to execute SQL queries");
        Preconditions.checkArgument((queryTimeout > 0 && queryTimeout <= this.maxQueryTimeout ? 1 : 0) != 0, (Object)"The timeout is not within the allowed range.\nRange: [1:3600]");
        ValidateCronExpression.validate(config.getSchedule());
        return this.saveRefreshConfiguration(tableName, queryTimeout, config);
    }

    private TestDataRefreshConfig saveRefreshConfiguration(@Nonnull String tableName, @Nonnull Integer queryTimeout, @Nonnull TestDataRefreshConfig config) {
        log.info("Saving refresh configuration for table: {}", (Object)tableName);
        List<TestDataTableCatalog> catalogList = this.getTableWithSameTitleAndQuery(tableName, config.isAllEnv());
        for (TestDataTableCatalog catalog : catalogList) {
            this.saveQueryTimeout(catalog.getTableName(), queryTimeout);
            config.setId(this.getConfigId(catalog));
            this.setRefreshConfig(catalog, config.getId());
            this.refreshConfigRepository.save(config);
            this.schedule(Collections.singletonList(config));
        }
        config.setQueryTimout(queryTimeout);
        log.info("Refresh configuration successfully saved.");
        return config;
    }

    private List<TestDataTableCatalog> getTableWithSameTitleAndQuery(String tableName, boolean allEnv) {
        TestDataTableCatalog tableCatalog = this.catalogRepository.findByTableName(tableName);
        if (!allEnv) {
            return Collections.singletonList(tableCatalog);
        }
        String tableQuery = this.importInfoRepository.findByTableName(tableCatalog.getTableName()).getTableQuery();
        List<TestDataTableCatalog> catalogList = this.catalogRepository.findAllByProjectIdAndTableTitle(tableCatalog.getProjectId(), tableCatalog.getTableTitle());
        List tableNameList = catalogList.stream().map(TestDataTableCatalog::getTableName).collect(Collectors.toList());
        List importInfoList = this.importInfoRepository.findAllById(tableNameList).stream().filter(importInfo -> importInfo.getTableQuery().equals(tableQuery)).map(TestDataTableImportInfo::getTableName).collect(Collectors.toList());
        return catalogList.stream().filter(c -> importInfoList.contains(c.getTableName())).collect(Collectors.toList());
    }

    private UUID getConfigId(TestDataTableCatalog tableCatalog) {
        if (Objects.nonNull(tableCatalog.getRefreshConfigId())) {
            return tableCatalog.getRefreshConfigId();
        }
        return UUID.randomUUID();
    }

    private void saveQueryTimeout(String tableName, Integer queryTimeout) {
        TestDataTableImportInfo importInfo = this.importInfoRepository.findByTableName(tableName);
        importInfo.setQueryTimeout(queryTimeout);
        this.importInfoRepository.save(importInfo);
    }

    private void setRefreshConfig(@Nonnull TestDataTableCatalog tableCatalog, @Nonnull UUID refreshConfigId) {
        tableCatalog.setRefreshConfigId(refreshConfigId);
        this.catalogRepository.save(tableCatalog);
    }

    @Override
    public RefreshResults runRefresh(@Nonnull UUID configId) {
        log.info("Run refresh data by using config with id: {}", (Object)configId);
        TestDataRefreshConfig config = this.getRefreshConfig(configId);
        TestDataTableCatalog catalog = this.catalogRepository.findByRefreshConfigId(config.getId());
        MdcUtils.put((String)MdcField.PROJECT_ID.toString(), (UUID)catalog.getProjectId());
        this.tdmMdcHelper.putConfigFields(catalog);
        this.metricService.executeRefreshJob(configId.toString(), catalog.getProjectId(), catalog.getTableTitle());
        RefreshResults results = new RefreshResults();
        if (config.isEnabled()) {
            String tableName = catalog.getTableName();
            log.info("Preparing to refresh. Table: {}", (Object)tableName);
            try {
                results = this.runRefresh(tableName, false);
            }
            catch (Exception e) {
                log.error("Error while executing refresh for table: {}", (Object)tableName, (Object)e);
            }
            log.info("Refresh data complete successful.");
            return results;
        }
        log.info("Refresh config not enabled.");
        return results;
    }

    @Override
    public List<RefreshResults> runRefresh(@Nonnull String tableName, @Nonnull Integer queryTimeout, @Nonnull boolean allEnv, boolean saveOccupiedData) throws Exception {
        ArrayList<RefreshResults> refreshResultsList = new ArrayList<RefreshResults>();
        List<TestDataTableCatalog> catalogList = this.getTableWithSameTitleAndQuery(tableName, allEnv);
        for (TestDataTableCatalog tableCatalog : catalogList) {
            this.testDataTableRepository.updateLastUsage(tableName);
            refreshResultsList.add(this.runRefresh(tableCatalog.getTableName(), saveOccupiedData));
        }
        return refreshResultsList;
    }

    @Override
    @Transactional
    public RefreshResults runRefresh(final @Nonnull String tableName, boolean saveOccupiedData) throws Exception {
        log.info("Run data refresh for table with name: {}, save occupied data: {}", (Object)tableName, (Object)saveOccupiedData);
        Server server = this.sqlRepository.getServer(tableName, this.catalogRepository, this.environmentsService);
        Optional importInfo = this.importInfoRepository.findById(tableName);
        if (!importInfo.isPresent()) {
            throw new RuntimeException("Import info not exist for table: " + tableName);
        }
        if (saveOccupiedData) {
            this.testDataTableRepository.deleteUnoccupiedRows(tableName);
        } else {
            this.testDataTableRepository.deleteAllRows(tableName);
        }
        Integer queryTimeout = (Integer)ObjectUtils.defaultIfNull((Object)((TestDataTableImportInfo)importInfo.get()).getQueryTimeout(), (Object)this.defaultQueryTimeout);
        JdbcTemplate userJdbcTemplate = this.sqlRepository.createJdbcTemplate(server, queryTimeout);
        final ArrayList columns = new ArrayList();
        final ArrayList<Map<String, Object>> rowsBuffer = new ArrayList<Map<String, Object>>();
        final int batchSize = 100;
        final AtomicReference<Integer> refreshedRows = new AtomicReference<Integer>(0);
        try {
            userJdbcTemplate.query(((TestDataTableImportInfo)importInfo.get()).getTableQuery(), new RowCallbackHandler(){

                public void processRow(ResultSet resultSet) throws SQLException {
                    if (columns.isEmpty()) {
                        ResultSetMetaData metaData = resultSet.getMetaData();
                        int columnCount = metaData.getColumnCount();
                        for (int columnIndex = 1; columnIndex <= columnCount; ++columnIndex) {
                            columns.add(metaData.getColumnName(columnIndex));
                        }
                    }
                    HashMap<String, Object> row = new HashMap<String, Object>();
                    for (String column : columns) {
                        row.put(column, resultSet.getObject(column));
                    }
                    rowsBuffer.add(row);
                    if (rowsBuffer.size() == batchSize) {
                        DataRefreshServiceImpl.this.testDataTableRepository.insertRows(tableName, true, rowsBuffer, (Integer)refreshedRows.get() > 0);
                        refreshedRows.updateAndGet(v -> v + batchSize);
                        rowsBuffer.clear();
                    }
                }
            });
        }
        catch (Exception e) {
            log.error("Error occurred while execute sql-request. %s", (Throwable)e);
            throw new TdmDbExecuteQueryException(e.getMessage());
        }
        if (!rowsBuffer.isEmpty()) {
            this.testDataTableRepository.insertRows(tableName, true, rowsBuffer, refreshedRows.get() > 0);
            refreshedRows.updateAndGet(v -> v + rowsBuffer.size());
        }
        if (refreshedRows.get() == 0) {
            throw new TdmSearchImportInfoException(tableName);
        }
        log.info("Total refreshed records: {}", (Object)refreshedRows.get());
        RefreshResults results = new RefreshResults();
        results.setRecordsTotal(refreshedRows.get());
        log.info("Data refresh has been finished");
        return results;
    }

    @Override
    public String getNextScheduledRun(String cronExpression) throws ParseException {
        CronExpression ce = new CronExpression(cronExpression);
        return ce.getNextValidTimeAfter(new Date()).toString();
    }

    public void removeUnused() {
        this.refreshConfigRepository.findAll().stream().map(TestDataRefreshConfig::getId).forEach(id -> {
            if (this.catalogRepository.findByRefreshConfigId((UUID)id).getTableTitle().isEmpty()) {
                this.refreshConfigRepository.deleteById(id);
                this.removeJob((UUID)id);
            }
        });
    }

    @Override
    public void removeJob(@Nonnull UUID configId) {
        log.info("Removing active refresh job for config with id: {}", (Object)configId);
        JobKey jobKey = new JobKey(configId.toString(), SCHED_GROUP);
        this.schedulerService.deleteJob(jobKey);
        log.info("Stored refresh job with id [{}] successfully removed.", (Object)configId);
    }

    private boolean checkSqlAvailability(@Nonnull String tableName) {
        Server server = this.sqlRepository.getServer(tableName, this.catalogRepository, this.environmentsService);
        this.sqlRepository.createJdbcTemplate(server);
        return true;
    }

    private void schedule(@Nonnull List<TestDataRefreshConfig> configs) {
        log.info("Processing [{}] refresh schedule request(s)", (Object)configs.size());
        for (TestDataRefreshConfig config : configs) {
            UUID configId = config.getId();
            log.info("Scheduling refresh config: {}", (Object)config.toString());
            JobDetail job = JobBuilder.newJob(DataRefreshJob.class).withIdentity(configId.toString(), SCHED_GROUP).build();
            this.schedulerService.reschedule(job, config, SCHED_GROUP);
        }
    }

    public void initSchedules() {
        log.info("Starting jobs for stored refresh configurations.");
        List<TestDataRefreshConfig> storedRequests = this.refreshConfigRepository.findAll().stream().filter(TestDataRefreshConfig::isEnabled).collect(Collectors.toList());
        this.schedule(storedRequests);
        log.info("Stored refresh jobs successfully started.");
    }
}

