/*
 * Decompiled with CFR 0.152.
 */
package org.qubership.profiler.dump;

import java.io.Closeable;
import java.io.File;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Queue;
import org.qubership.profiler.dump.DumpFile;
import org.qubership.profiler.dump.DumpFileLog;
import org.qubership.profiler.dump.DumpFilesFinder;
import org.qubership.profiler.dump.DumpRoot;
import org.qubership.profiler.dump.FileDeleter;
import org.qubership.profiler.io.listener.FileRotatedListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DumpFileManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(DumpFileManager.class);
    private final FileRotatedListener fileRotatedListener = new FileRotatedListener(){

        @Override
        public void fileRotated(DumpFile oldFile, DumpFile newFile) {
            if (oldFile == null) {
                return;
            }
            try {
                DumpFileManager.this.registerFile(oldFile, true);
            }
            catch (RuntimeException e) {
                log.warn("Can't register file {}", (Object)oldFile, (Object)e);
            }
        }
    };
    private final Map<String, DumpRoot> dumpRootsOrdered = new LinkedHashMap<String, DumpRoot>();
    private long maxAgeMillis;
    private long maxSize;
    private String root;
    private boolean initialized = false;
    private final FileDeleter fileDeleter = new FileDeleter();
    private DumpFileLog dumpFileLog;

    public DumpFileManager(long maxAgeMillis, long maxSize, String root) {
        this(maxAgeMillis, maxSize, root, false);
    }

    public DumpFileManager(long maxAgeMillis, long maxSize, String root, boolean forceScanFolder) {
        try {
            this.init(maxAgeMillis, maxSize, root, forceScanFolder);
        }
        catch (RuntimeException e) {
            log.error(String.format("Can't initialize DumpFileManager with given parameters (maxAgeMillis: %d, maxSize: %d, root directory: %s)", maxAgeMillis, maxSize, root), e);
        }
    }

    public synchronized void rescan() {
        this.close();
        this.init(this.maxAgeMillis, this.maxSize, this.root, true);
    }

    private void init(long maxAgeMillis, long maxSize, String root, boolean forceScanFolder) {
        if (this.initialized) {
            throw new IllegalStateException("DumpFileManager is already initialized. Close before reinitializing");
        }
        log.info("Initialize DumpFileManager with maxSize = {}, maxAgeMillis = {}, root = {}", maxSize, maxAgeMillis, root);
        this.maxAgeMillis = maxAgeMillis;
        this.maxSize = maxSize;
        this.root = root;
        this.dumpRootsOrdered.clear();
        File fileList = new File(root, "filelist.blst");
        boolean fileListParsed = false;
        this.dumpFileLog = new DumpFileLog(fileList);
        if (!forceScanFolder) {
            try {
                Queue<DumpFile> queue = this.dumpFileLog.parseIfPresent();
                if (queue != null) {
                    log.info("Found dump files list in {}", (Object)fileList);
                    for (DumpFile dumpFile : queue) {
                        this.registerFile(dumpFile, false, false);
                    }
                    fileListParsed = true;
                }
            }
            catch (RuntimeException e) {
                log.warn(String.format("Can't parse file %s. It will be rewritten", fileList), e);
            }
        }
        if (!fileListParsed) {
            if (forceScanFolder) {
                log.info("DumpFileManager is forced to read dump directory");
            }
            log.info("Read dump root directory {}", (Object)root);
            this.dumpFileLog.cleanup(null, true);
            DumpFilesFinder dumpFilesFinder = new DumpFilesFinder();
            Queue<DumpFile> dumpFiles = dumpFilesFinder.search(this.root);
            for (DumpFile dumpFile : dumpFiles) {
                this.registerFile(dumpFile, false);
            }
        }
        this.tryPruneOldFiles();
        this.initialized = true;
    }

    public void registerFile(DumpFile file, boolean tryPrune) {
        this.registerFile(file, tryPrune, true);
    }

    public void registerFile(DumpFile file, boolean tryPrune, boolean writeLog) {
        this.addFileWithoutStoring(file);
        if (writeLog) {
            this.dumpFileLog.writeAddition(file);
        }
        log.info("Added info about dump file {}. Current size is {} bytes", (Object)file, (Object)this.getCurrentSize());
        if (tryPrune) {
            this.tryPruneOldFiles();
        }
    }

    public long getCurrentSize() {
        long res = 0L;
        for (DumpRoot dumpRoot : this.dumpRootsOrdered.values()) {
            res += dumpRoot.getSizeInBytes();
        }
        return res;
    }

    private synchronized void addFileWithoutStoring(DumpFile file) {
        DumpRoot dumpRoot = this.dumpRootsOrdered.get(file.getDumpRootName());
        if (dumpRoot == null) {
            File f = new File(file.getPath());
            File typeDir = f.getParentFile();
            File dumpRootDir = typeDir.getParentFile();
            String dumpRootPath = dumpRootDir.getPath();
            dumpRoot = new DumpRoot(dumpRootPath);
            this.dumpRootsOrdered.put(dumpRoot.getName(), dumpRoot);
        }
        dumpRoot.registerFile(file);
    }

    private DumpFile getFirstFile() {
        if (this.dumpRootsOrdered.isEmpty()) {
            return null;
        }
        for (DumpRoot dumpRoot : this.dumpRootsOrdered.values()) {
            DumpFile file = dumpRoot.getFirstFile();
            if (file == null) continue;
            return file;
        }
        return null;
    }

    public synchronized void tryPruneOldFiles() {
        long currentSize;
        long deletedSize;
        if (this.dumpRootsOrdered.isEmpty()) {
            log.info("No files registered. Nothing to delete on pruning. Will wait for the next rotating");
            return;
        }
        log.info("Try prune dump files. Max age: {}; max size: {}", (Object)this.maxAgeMillis, (Object)this.maxSize);
        if (this.maxAgeMillis != 0L) {
            DumpFile elderFile = this.getFirstFile();
            long borderTimestamp = System.currentTimeMillis() - this.maxAgeMillis;
            while (elderFile != null && elderFile.getTimestamp() < borderTimestamp) {
                deletedSize = this.deleteDumpFile(elderFile);
                elderFile = this.getFirstFile();
            }
        }
        for (currentSize = this.getCurrentSize(); this.maxSize != 0L && currentSize > this.maxSize; currentSize -= deletedSize) {
            DumpFile fileToDelete = this.getFirstFile();
            if (fileToDelete == null) {
                log.error("There are no registered files. But current size should be {}. Try 'Rescan' button from the UI", (Object)currentSize);
                return;
            }
            deletedSize = this.deleteDumpFile(fileToDelete);
        }
        log.info("Dump files size after pruning is {}", (Object)currentSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long deleteDumpFile(DumpFile file) {
        log.info("Deleting {}", (Object)file);
        DumpFileManager dumpFileManager = this;
        synchronized (dumpFileManager) {
            DumpRoot dumpRoot = this.dumpRootsOrdered.get(file.getDumpRootName());
            long sizeBefore = dumpRoot.getSizeInBytes();
            Collection<DumpFile> deletedFiles = dumpRoot.deleteFile(file, this.fileDeleter);
            long sizeAfter = dumpRoot.getSizeInBytes();
            for (DumpFile deletedFile : deletedFiles) {
                this.dumpFileLog.writeDeletion(deletedFile);
            }
            if (dumpRoot.isEmpty()) {
                this.dumpRootsOrdered.remove(dumpRoot.getName());
            }
            return sizeBefore - sizeAfter;
        }
    }

    private boolean tryDeleteByModifiedDate(DumpFile file) {
        if (file.getDependentFile() != null) {
            return false;
        }
        if (0L == this.maxAgeMillis) {
            return false;
        }
        long borderModificationTime = System.currentTimeMillis() - this.maxAgeMillis;
        if (file.getTimestamp() < borderModificationTime) {
            log.info("File {} has modification time {} which is older than allowable {}. Will delete it", file, file.getTimestamp(), borderModificationTime);
            boolean success = this.fileDeleter.deleteFile(file);
            this.dumpFileLog.writeDeletion(file);
            return success;
        }
        return false;
    }

    public FileRotatedListener getFileRotatedListener() {
        return this.fileRotatedListener;
    }

    @Override
    public void close() {
        this.dumpFileLog.close();
        this.initialized = false;
    }
}

