; This Source Code Form is subject to the terms of the Mozilla Public License,
; v. 2.0. If a copy of the MPL was not distributed with this file, You can
; obtain one at http://mozilla.org/MPL/2.0/.

(ns quilt.sedan.impl
  (:require [quilt.sedan.numbers :as number
             :refer (==-0 integer? float? number? finite? zero? neg? decimal?
                          ncompare bigdec bigint scale)]
            [quilt.sedan.sbuffers :refer (Writable write)]
            [clojure.string :as str])
  (                      :require-macros
         [quilt.sedan.macros :refer (!0 define-partition!)])
  (:refer-clojure :exclude (integer? float? number? zero? neg?
                                                 
                                                 
                                                   )))

(defprotocol Sedan
  "Internal implementation protocol for Sedan. You generally shouldn't need to
  use this at all, see `encode`, `compare`, `bottom`, and `top` in
  the quilt.sedan ns."
  (encode* [x buffer])
  (tag-for [x]
  ; NOTE tags are [0-127] so they UTF-8 encode to a single byte
  "Returns the type tag of [x] (a value, not a class or prototype).
  Will be an int [0-127].")
  (compare* [a b]
    "Similar to `clojure.core/compare`, but implements at runtime
     the same collation semantics that `partition` and `encode*`
     produce in Sedan-encoded data."))

(def ^:private partitions
  "Map from tag bytes/chars/single-char strings to map of each corresponding
partition's :name (largely just for doc / error messages) and a :decoder
function for that partition's type."
  {})

(defn register-partition!
  [tag-byte name decoder-fn]
  (let [partition-info {:partition name
                        :decoder decoder-fn}
        partition-pairs (map #(update-in % [0] %2)
                             (repeat [tag-byte partition-info])
                             [identity char (comp str char)])]
                                                            
           (set! partitions (into partitions partition-pairs))))

(defn without-tag
  [s]
  (subs s 1))

(defn encode
  [x buffer]
  (write buffer (char (tag-for x)))
  (encode* x buffer))

(defn decode [^String s]
  (let [tag (                          .charCodeAt s 0)
        decoder (:decoder (partitions tag))]
    (assert decoder (str "No decoder available for partition "
                         (-> tag partitions :name) ", tag " (int tag)))
    (decoder tag s)))

(defn default-compare
  [a b]
  (!0 (compare (tag-for a) (tag-for b))
      (compare a b)))

; records only to get sane equality/hashing; maybe make them types at a later date
(defrecord Bottom []
                          IComparable
  (                       -compare [a b]
    (if (instance? Bottom b)
      0
      -1)))

(defrecord Top []
                          IComparable
  (                       -compare [a b]
    (if (instance? Top b)
      0
      1)))

     
   
                           
                            
                                
                                   
         
                                              

; starting at 0x02 to minimize escaping of encoded values in lists
(define-partition! 0x02 :bottom Bottom
  (fn decode-bottom [tag s] (Bottom.))
  (encode* [_ _])
  (compare* [a b] (default-compare a b)))

(define-partition! 0x7f :top Top
  (fn decode-top [tag s] (Top.))
  (encode* [_ _])
  (compare* [a b] (default-compare a b)))

; /ht bytewise for encoding / separator scheme in sequence types
; we could either read/write char-by-char, or regex; latter is two passes, but
; simpler and perhaps still faster, esp. on JS where we'd otherwise be doing a
; ton of single-char string appends
(defn- escaping-buffer
  [sb]
  (reify Writable
    (write [_ string]
      (write sb (-> (str/replace string #"\u0001" "\u0001\u0002")
                    (str/replace #"\u0000" "\u0001\u0001"))))))

(defn- unescape
  [string]
  (-> (str/replace string #"\u0001\u0001" "\u0000")
      (str/replace #"\u0001\u0002" "\u0001")))

(defn encode-sequence
  [xs buffer prefix-count?]
  ; this count is strictly here to force collation of encoded sequences on
  ; relative length first, then contents in turn...not used in decoding at all
  (when prefix-count?
    (write buffer (number/encode-int (count xs)))
    (write buffer "\u0000"))
  
  (let [esc-buffer (escaping-buffer buffer)
        between-marker                        #js {}]
    (doseq [x (interpose between-marker xs)]
      (if (identical? x between-marker)
        (write buffer "\u0000")
        (encode x esc-buffer)))))

(defn decode-sequence
  [ctor ^String s drop-first?]
  (->> (drop (if drop-first? 1 0) (.split s "\u0000"))
       (map unescape)
       (map decode)
       (apply ctor)))

(defn- compare-sequential
  [a b]
  (!0 (compare (tag-for a) (tag-for b))
      (compare (count a) (count b))
      (or (->> (map compare* a b)
           (filter #(not (zero? %)))
           first)
          0)))

(defn- encode-namespaced
  [n buffer]
  (let [esc-buffer (escaping-buffer buffer)]
    (write esc-buffer (namespace n))
    (write buffer "\u0000")
    (write esc-buffer (name n))))

(defn- decode-namespaced
  [ctor ^String s]
  ; not encoding components here like we do for sequences, so they can be
  ; zero-length, need to pad if a kw/symbol had an empty namespace or name
  (let [split (.indexOf s "\u0000")
        ns (subs s 0 split)
        name (subs s (inc split))]
    (ctor (unescape ns) (unescape name))))

(define-partition! 0x03 :nil nil
  (fn decode-nil [tag s] nil)
  (encode* [_ _])
  (compare* [a b] (default-compare a b)))

(define-partition! #{0x04 0x05} :boolean                      boolean
  (fn decode-boolean [tag s]
    (case tag
      0x04 false
      0x05 true))
  (encode* [x buffer])
  (tag-for [x] (if x 0x05 0x04))
  (compare* [a b] (default-compare a b)))

; incredible complexity here because of the infinities, -0.0, and exponential
; bigdec zeros. Clever alternatives welcome.
(defn- compare-numbers
  [a b]
  (or (and (zero? a) (zero? b)
           ; prior failure #7
           (let [ba? (decimal? a)
                 bb? (decimal? b)]
             (cond
              (and ba? bb?) (- (compare (scale a) (scale b)))
              ba? (if (==-0 b)
                    1
                    (compare (- (scale a)) 0))
              bb? (if (==-0 a)
                    -1
                    (compare 0 (- (scale b))))
              :else nil)))

      (let [fa? (float? a)
            fb? (float? b)]
        (cond
         (and fa? fb?)
         (cond
          (==-0 a) (cond (==-0 b) 0 (neg? b) 1 :else -1)
          (==-0 b) (if (neg? a) -1 1)
          :else (ncompare a b))

         fa?
         (cond
          (==-0 a) (if (neg? b) 1 -1)
          ; prior failure #1
          (finite? a) (ncompare (bigdec a) b)
          :else (if (== a number/-INF) -1 1))

         fb?
         (cond
          (==-0 b) (if (neg? a) -1 1)
          ; prior failure #1
          (finite? b) (ncompare a (bigdec b))
          :else (if (== b number/-INF) 1 -1))

         :else (ncompare a b)))))

(let [number-decode (fn decode-number [tag s] (number/decode s))]
  (define-partition! #{0x10 0x11 0x12 0x13 0x14 0x15 0x16 0x17} :number
                        #{number/BigInteger number/BigDecimal number}
    number-decode
    (tag-for [x] (number/tag-for x))
    (encode* [x buffer]
      (write buffer
        ; so stupid I want to spit
        ; TODO Sedan impl protocol broken; need to let partition handlers
        ; completely control encoding sometimes; same as with booleans, but that
        ; was so trivial it didn't seem important originally
        (subs (number/encode x) 1)))
    (compare* [a b]
      (if (number? b)
        (compare-numbers a b)
        (default-compare a b)))))

(define-partition! 0x20 :string                     string
  (fn decode-string [tag s] (without-tag s))
  (encode* [x buffer] (write buffer x))
  (compare* [a b] (default-compare a b)))

(define-partition! #{0x27 0x28} :symbol                                  cljs.core/Symbol
  (fn decode-symbol [tag s]
    (if (== tag 0x27)
      (symbol (without-tag s))
      (decode-namespaced symbol (without-tag s))))
  (tag-for [x] (if (namespace x) 0x28 0x27))
  (encode* [x buffer]
    (if (namespace x)
      (encode-namespaced x buffer)
      (write buffer (name x))))
  (compare* [a b] (default-compare a b)))

(define-partition! #{0x29 0x2a} :keyword                                   cljs.core/Keyword
  (fn decode-keyword [tag s]
    (if (== tag 0x29)
      (keyword (without-tag s))
      (decode-namespaced keyword (without-tag s))))
  (tag-for [x] (if (namespace x) 0x2a 0x29))
  (encode* [x buffer]
    (if (namespace x)
      (encode-namespaced x buffer)
      (write buffer (name x))))
  (compare* [a b] (default-compare a b)))

(define-partition! 0x30 :instant                             js/Date
  (fn decode-instant [tag s]
    (                             js/Date. (number/decode-int (without-tag s))))
  (encode* [x buffer] (write buffer (number/encode-int (.getTime x))))
  (compare* [a b] (default-compare a b)))

(define-partition! 0x31 :uuid                             cljs.core/UUID
  (fn decode-uuid [tag s]
    (                                       cljs.core/UUID. (without-tag s)))
  ; UUIDs on the JVM compare based on their component long values -- which don't
  ; exist in ClojureScript. Also, insofar as sort order of UUIDs is effectively
  ; arbitrary, there seems to be no good reason to use the string representation
  ; of the UUID for sorting (normalized to lower case since default string
  ; representations aren't guaranteed one way or the other).
  (encode* [x buffer]
    (write buffer (str/lower-case                      (.-uuid x))))
  (compare* [a b]
    (!0 (compare (tag-for a) (tag-for b))
        (compare (str/lower-case                      (.-uuid a))
                 (str/lower-case                      (.-uuid b))))))

(defn- install-impl-for
  [x]
  (let [type (type x)]
    (cond
     (sequential? x) (extend-type                              type
                                          Sedan
                                          (tag-for [_] 0x60)
                                          (encode* [x buffer] (encode-sequence x buffer true))
                                          (compare* [a b] (compare-sequential a b)))

     :default (throw (ex-info (str "No way to encode value of type " type)
                              {:value x
                               :type type})))))

(register-partition! 0x60 :sequential
  (fn decode-list [tag s] (decode-sequence vector (without-tag s) true)))

(extend-protocol Sedan
                      object
  (tag-for [x]
    (install-impl-for x)
    (tag-for x))
  ; encode* call is irrelevant, should only ever be called after tag-for
  ; decode* will never be called
  (compare* [a b]
    (install-impl-for a)
    (compare* a b)))

;; TODO add js bignumber dispatch

;;;;;;;;;;;; This file autogenerated from src/cljx/quilt/sedan/impl.cljx
