package org.ninjax.core;

import com.google.common.collect.ImmutableMap;
import io.jsonwebtoken.Jwts;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.Optional;
import javax.crypto.SecretKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NinjaSessionConverter {

    private static final Logger logger = LoggerFactory.getLogger(NinjaSessionConverter.class);

    public static final String NINJA_SESSION_COOKIE_NAME = "NINJA_SESSION";
    private static final String NINJA_SESSION_PATH = "/";

    public static Optional<NinjaSession> extractSessionFromCookie(
            NinjaCookie ninjaSessionCookie,
            SecretKey secretKeyForSessionEncryption) {

        var now = System.currentTimeMillis();

        try {
            var claims = Jwts.parser()
                    .verifyWith(secretKeyForSessionEncryption)
                    .build()
                    .parseSignedClaims(ninjaSessionCookie.value())
                    .getPayload();

            if (claims.getNotBefore() != null /* Not our Api. We have to do a null check :( */
                    && now < claims.getNotBefore().getTime()) {
                return Optional.empty();
            }

            if (claims.getExpiration() != null /* Not our Api. We have to do a null check :( */
                    && now > claims.getExpiration().getTime()) {
                return Optional.empty();
            }

            var mapBuilder = ImmutableMap.<String, String>builder();
            for (var e : claims.entrySet()) {
                mapBuilder.put(e.getKey(), e.getValue().toString());
            }
            var ninjaSession = new NinjaSession(mapBuilder.build());

            return Optional.of(ninjaSession);
        } catch (Exception e) {
            logger.debug("Opsi. Error parsing Ninja Session. I am ignoring this session.", e);
            return Optional.empty();
        }

    }

    public static NinjaCookie createCookieToRemoveNinjaSession() {
        int REMOVE_SESSION_MAX_AGE = 0;
        var cookie = new NinjaCookie(
                NINJA_SESSION_COOKIE_NAME,
                "",
                Optional.empty(),
                Optional.empty(),
                REMOVE_SESSION_MAX_AGE,
                Optional.of(NINJA_SESSION_PATH),
                false,
                false);

        return cookie;
    }

    public static NinjaCookie createCookieWithInformationOfNinjaSession(
            NinjaSession ninjaSession,
            SecretKey secretKeyForSessionEncryption,
            Optional<Long> sessionExpiryTimeInSeconds) {

        // some setup
        Instant now = Instant.now();

        Optional<Instant> expiryInstant = Optional.empty();

        if (ninjaSession.get("exp").isPresent()) {
            expiryInstant = Optional.of(Instant.ofEpochSecond(Long.parseLong(ninjaSession.get("exp").get())));
        }

        if (expiryInstant.isEmpty() && sessionExpiryTimeInSeconds.isPresent()) {
            expiryInstant = Optional.of(now.plusSeconds(sessionExpiryTimeInSeconds.get()));
        }

        // build jwt
        var nowDate = Date.from(now);
        var jwsBuilder = Jwts.builder()
                .notBefore(nowDate)
                .issuedAt(nowDate);

        expiryInstant.ifPresent(i -> jwsBuilder.expiration(Date.from(i)));

        String jws = jwsBuilder
                .claims(ninjaSession.keyValueStore())
                .signWith(secretKeyForSessionEncryption)
                .compact();

        var maxAge = expiryInstant.map(i -> (int) Duration.between(now, i).getSeconds())
                .orElse(0); // 0 is a session cookie

        //build cookie from jwt
        var cookie = new NinjaCookie(
                NINJA_SESSION_COOKIE_NAME,
                jws,
                Optional.empty(),
                Optional.empty(),
                maxAge,
                Optional.of(NINJA_SESSION_PATH),
                false,
                false);

        return cookie;
    }

}
