package org.readutf.hermes.channel

import com.github.michaelbull.result.Err
import com.github.michaelbull.result.Ok
import com.github.michaelbull.result.Result
import io.github.oshai.kotlinlogging.KotlinLogging
import org.readutf.hermes.Packet
import org.readutf.hermes.PacketManager
import org.readutf.hermes.response.ResponsePacket
import java.util.concurrent.CompletableFuture

abstract class HermesChannel(
    val channelId: String,
    val packetManager: PacketManager<*>,
) {
    val logger = KotlinLogging.logger {}

    abstract fun sendPacket(packet: Packet): Result<Unit, Throwable>

    abstract fun close()

    inline fun <reified T> sendPacketFuture(packet: Packet): CompletableFuture<Result<T, String>> {
        val storedFuture = CompletableFuture<ResponsePacket>()

        logger.debug { "storing ${packet.packetId}" }

        packetManager.responseFutures[packet.packetId] = storedFuture
        sendPacket(packet)

        var packetFuture = CompletableFuture<T>()

        return storedFuture.thenApply { responsePacket ->
            logger.debug { "Received response back for packet ${packet.packetId}" }

            if (responsePacket.response is T) {
                logger.debug { "Received back $responsePacket as ${T::class.java.simpleName}" }
                if (responsePacket.success) {
                    return@thenApply Ok(responsePacket.response as T)
                } else {
                    return@thenApply Err(responsePacket.error)
                }
            } else {
                logger.warn { "Response packet was not of type ${T::class.java.simpleName}" }
                throw IllegalStateException("Response packet was not of type ${T::class.java.simpleName}")
            }
        }
    }
}
