/*
 * RecommenderNeo4jEngineManager.java
 * 
 * Copyright (C) 2013 Alessandro Negro <alessandro.negro at reco4j.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.reco4j.graph.neo4j.engine;

import org.reco4j.model.Rating;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.configuration.Configuration;
import org.neo4j.graphdb.DynamicRelationshipType;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.reco4j.graph.*;
import org.reco4j.engine.RecommenderBuilderThread;
import org.reco4j.engine.RecommenderEngine;
import org.reco4j.graph.neo4j.Neo4JEdge;
import org.reco4j.graph.neo4j.Neo4JNode;
import org.reco4j.graph.neo4j.Neo4jGraph;
import org.reco4j.graph.neo4j.util.Neo4JPropertiesHandle;
import org.reco4j.recommender.IRecommender;

/**
 *
 * @author Alessandro Negro <alessandro.negro at reco4j.org>
 */
public class RecommenderNeo4jEngineManager
{
  private static final Logger logger = Logger.getLogger(RecommenderNeo4jEngineManager.class.getName());
  private Neo4jGraph learningDataSet;
  private IRecommender recommender;
  private RecommenderBuilderThread recommenderThread;

  private RecommenderNeo4jEngineManager()
  {
  }

  public void setup(GraphDatabaseService gds)
  {
    learningDataSet = new Neo4jGraph(Neo4JPropertiesHandle.getInstance());
    learningDataSet.setDatabase(gds);
  }

  public static RecommenderNeo4jEngineManager getInstance()
  {
    return RecommenderNeo4jEngeManagerHolder.INSTANCE;
  }

  private Properties loadProperties(String propertiesPath)
  {
    Properties properties = new Properties();
    try
    {
      properties.load(new FileReader(propertiesPath));
    }
    catch (IOException ex)
    {
      Logger.getLogger(RecommenderNeo4jEngineManager.class.getName()).log(Level.SEVERE, "Error while loading properties: " + propertiesPath, ex);
    }
    return properties;
  }

  public Iterable<Node> getRecommender(INode user, int type)
  {
    List<Rating> recommendations = recommender.recommend(user);
    ArrayList<Node> result = new ArrayList<Node>();
    for (Rating reco : recommendations)
      result.add(((Neo4JNode) reco.getItem()).getNode());
    logger.log(Level.INFO, "Size {0}", result.size());
    return result;
  }

  public void updateRecommender(Relationship edge, int operation)
  {
    if (edge.isType(DynamicRelationshipType.withName(EdgeTypeFactory.getEdgeType(IEdgeType.EDGE_TYPE_RANK, recommender.getConfig().getGraphConfig()).getEdgeName())))
      recommender.updateRecommender(new Neo4JEdge(edge), operation);
  }

  private static class RecommenderNeo4jEngeManagerHolder
  {
    private static final RecommenderNeo4jEngineManager INSTANCE = new RecommenderNeo4jEngineManager();
  }

  public void init(GraphDatabaseService gds, Configuration c)
  {
    setup(gds);

    String propertiesPath = c.getString("org.reco4j.graph.neo4j.engine.properties");
    Properties properties = loadProperties(propertiesPath);
    recommender = RecommenderEngine.createRecommender(properties);
    recommenderThread = new RecommenderBuilderThread(recommender, learningDataSet);
    logger.log(Level.INFO, "Reco4j Initialized!");
  }

  public void startRecommender()
  {
    try
    {
      recommenderThread.start();
    }
    catch (Exception ex)
    {
      logger.log(Level.SEVERE, "Error in recommender Thread!!!", ex);
    }
  }

  public Iterable<Node> getUsers()
  {
    List<INode> users = learningDataSet.getUserNodes();
    ArrayList<Node> result = new ArrayList<Node>();
    for (INode node : users)
      result.add(((Neo4JNode) node).getNode());
    return result;
  }

  public Iterable<Node> getItems()
  {
    List<INode> items = learningDataSet.getItemNodes();
    ArrayList<Node> result = new ArrayList<Node>();
    for (INode node : items)
      result.add(((Neo4JNode) node).getNode());
    return result;
  }
  
  public Iterable<Relationship> getAllRatings()
  {
    List<IEdge> rels = learningDataSet.getEdgesByType(EdgeTypeFactory.getEdgeType(IEdgeType.EDGE_TYPE_RANK, recommender.getConfig().getGraphConfig()));
    ArrayList<Relationship> result = new ArrayList<Relationship>();
    for (IEdge item : rels)
      result.add(((Neo4JEdge) item).getEdge());
    return result;
  }

  public Iterable<Relationship> getRatings(Node node)
  {
    Neo4JNode n = new Neo4JNode(node);
    List<IEdge> relationships = n.getOutEdge(EdgeTypeFactory.getEdgeType(IEdgeType.EDGE_TYPE_RANK, recommender.getConfig().getGraphConfig()));
    ArrayList<Relationship> result = new ArrayList<Relationship>();
    for (IEdge relationship : relationships)
      result.add(((Neo4JEdge) relationship).getEdge());
    return result;
  }

  public Iterable<Relationship> getRatingsTest(Node node)
  {
    Neo4JNode n = new Neo4JNode(node);
    List<IEdge> relationships = n.getOutEdge(EdgeTypeFactory.getEdgeType(IEdgeType.EDGE_TYPE_TEST_RANK, recommender.getConfig().getGraphConfig()));
    ArrayList<Relationship> result = new ArrayList<Relationship>();
    for (IEdge relationship : relationships)
      result.add(((Neo4JEdge) relationship).getEdge());
    return result;
  }
}
