
/**
 * Copyright 2011-2012 Clint Combs
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.regrest

import org.regrest.http.RequestProcessor
import org.regrest.restclient.RequestClientRequestSource
import java.io.File
import org.regrest.restclient.RestClientRunner
import org.regrest.report.{HtmlReporter,PlainTextReporter,ReportFormat}
import org.regrest.report.ReportFormat._
import scopt.OptionParser
import com.weiglewilczek.slf4s.Logging

/**
 * Regrest is an App for regression testing.  It loads REST requests, sends them
 * to the appropriate resources, and reports the results.
 */
object Regrest extends App with Logging {

	var config:RegrestConfig = defaultConfig

	val parser = new OptionParser("org.regrest.Regrest") {
		opt("oh", "overrideHost", "<host>",
			"override the host used for each request",
			{ host:String => config.overrideHost = Some(host) })
		arg("<test-script>", "a test script to run on start-up",
			{ v:String => config.testScript = Some(v) })
	}

	if (parser.parse(args)) run

	def run = {
		logger.info(config.toString)
		val wd = config.workingDirectory match {
			case Some(s) => Some(new File(s))
			case None => logger.error("workingDirectory not set"); None
		}
		if (wd != None) {
			config.reportFile match {
				case Some(f) =>
					if (config.reportFileFormat.get == html) renderHtml(f, wd.get) else renderText(f, wd.get)
				case None =>
					logger.error("report file not specified")
			}
		}
	}

	def renderHtml(reportFileName:String, workingDirectory:File) = {
		new RestClientRunner(
			inputFileOrDirectory=config.inputFileOrDirectory,
			reportFile=new File(reportFileName),
			runnerWorkingDirectory=workingDirectory,
			overrideHost=config.overrideHost) with HtmlReporter run
	}

	def renderText(reportFileName:String, workingDirectory:File) = {
		new RestClientRunner(
			inputFileOrDirectory=config.inputFileOrDirectory,
			reportFile=new File(reportFileName),
			runnerWorkingDirectory=workingDirectory,
			overrideHost=config.overrideHost) with PlainTextReporter run
	}

	def defaultConfig():RegrestConfig = {
		val cfg = RegrestConfig()

		val regrestDataDir = new File(System.getProperty("user.home"), "regrest-data")

		if (!regrestDataDir.exists) regrestDataDir.mkdir

		cfg.testsDir = Some(new File(regrestDataDir, "tests"))
		if (!cfg.testsDir.get.exists) cfg.testsDir.get.mkdir

		cfg.resultsDir = Some(new File(regrestDataDir, "results"))
		if (!cfg.resultsDir.get.exists) cfg.resultsDir.get.mkdir

		cfg.workingDir = Some(new File(regrestDataDir, "working"))
		if (!cfg.workingDir.get.exists) cfg.workingDir.get.mkdir

		cfg.reportFile = Option(new File(cfg.resultsDir.get, "output.txt").getAbsolutePath)
		cfg.workingDirectory = Option(cfg.workingDir.get.getAbsolutePath)

		cfg
	}
}

case class RegrestConfig(
	var testsDir:Option[File] = None,
	var resultsDir:Option[File] = None,
	var workingDir:Option[File] = None,
	var testScript:Option[String] = None,
	var reportFile:Option[String] = None,
	var reportFileFormat:Option[ReportFormat.Value] = Option(text),
	var workingDirectory:Option[String] = None,
	var overrideHost:Option[String] = None) {

	def inputFileOrDirectory:File = new File(testsDir.get, testScript.get + ".rcq")
}
