
/**
 * Copyright 2011-2012 Clint Combs
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.regrest.restclient

import org.regrest.RestRequest
import java.io.{File,FileFilter}
import com.weiglewilczek.slf4s.Logging

class RequestClientRequestSource(fileOrDirectory:File) extends Logging {

	def loadRestRequests():List[(File, Option[RestRequest])] = {
		val loader = new RestClientFileLoader();
		val fileAndRestRequestOptions =
			for(f <- gatherFiles(fileOrDirectory)) yield (f, loader.loadRestClientRequest(f))
		printFileMessages(fileOrDirectory, fileAndRestRequestOptions)
		fileAndRestRequestOptions
	}

	def printFileMessages(fileOrDirectory:File, fileXmlOptions:List[(File, Option[RestRequest])]) = {
		logger.info("attempting to load requests from " + fileXmlOptions.length + " file"
			+ (if (fileXmlOptions.length > 1) "s found in " else " ") + fileOrDirectory)

		val filesWithoutAlerts = (0 /: fileXmlOptions) { (r, s) =>
			r + (if (s._2.isEmpty) 1 else 0)
		}
		if (filesWithoutAlerts == 1) logger.info("1 file does not contain a request")
			else if (filesWithoutAlerts > 1)
				logger.info(filesWithoutAlerts + " files do not contain requests")

		val requestCount = fileXmlOptions.length-filesWithoutAlerts
		logger.info("sending " + requestCount + " request" + (if (requestCount > 1) "s" else ""))
	}

	def gatherFiles(fileOrDirectory:File):List[File] = {
		if (fileOrDirectory.isDirectory) {
			(List[File]() /: fileOrDirectory.listFiles(new RestClientFileFilter())) {
				(l, i) => i :: l
			} reverse
		} else List[File](fileOrDirectory)
	}
}

/**
 * Filter for RestClient .rcq and .xml files.
 */
class RestClientFileFilter extends FileFilter {
	def accept(f:File) = f.isFile && f.getName.endsWith(".rcq")
}
