/*
 * TeleStax, Open Source Cloud Communications
 * Copyright 2011-2016, Telestax Inc and individual contributors
 * by the @authors tag.
 *
 * This program is free software: you can redistribute it and/or modify
 * under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 */
package org.restcomm.commons.statistics.reporter;

import com.codahale.metrics.*;
import com.codahale.metrics.Timer;

import java.util.*;
import java.util.concurrent.TimeUnit;
import org.restcomm.commons.statistics.sender.RestcommStatsSender;

/**
 * This class represents a reporter that implements persistence (into the central server) for statistical events
 * generated by modules of restcomm platform.
 *
 * @author Ricardo Limonta
 * @since 2016-05-29
 */
public class RestcommStatsReporter extends ScheduledReporter {

    private final Clock clock;
    private Map<String, Object> values;
    private String remoteServer;

    public RestcommStatsReporter(MetricRegistry registry,
            MetricFilter filter,
            TimeUnit rateUnit,
            TimeUnit durationUnit,
            Clock clock) {
        super(registry, "restcomm-stats-reporter", filter, rateUnit, durationUnit);
        this.clock = clock;
    }
    
    public static Builder forRegistry(MetricRegistry registry) {
        return new Builder(registry);
    }
    
    @Override
    public void report(SortedMap<String, Gauge> gauges,
            SortedMap<String, Counter> counters,
            SortedMap<String, Histogram> histograms,
            SortedMap<String, Meter> meters,
            SortedMap<String, Timer> timers) {

        final long timestamp = clock.getTime();

        
        //Process Gauge Metrics
        for (Map.Entry<String, Gauge> gauge : gauges.entrySet()) {
            values = new HashMap<>();
            values.put("timestamp", timestamp);
            values.put("key", gauge.getKey());
            values.put("value", gauge.getValue().getValue());
            RestcommStatsSender.sendStats(values, "gauge", this.remoteServer);
        }

        //Process Counter Metrics
        for (Map.Entry<String, Counter> counter : counters.entrySet()) {
            values = new HashMap<>();
            values.put("timestamp", timestamp);
            values.put("key", counter.getKey());
            values.put("count", counter.getValue().getCount());
            RestcommStatsSender.sendStats(values, "counter", this.remoteServer);
        }

        //Process Histogram Metrics
        for (Map.Entry<String, Histogram> hist : histograms.entrySet()) {
            values = new HashMap<>();
            values.put("timestamp", timestamp);
            values.put("key", hist.getKey());
            values.put("count", hist.getValue().getCount());
            values.put("size", hist.getValue().getSnapshot().size());
            values.put("max", hist.getValue().getSnapshot().getMax());
            values.put("mean", hist.getValue().getSnapshot().getMean());
            values.put("media", hist.getValue().getSnapshot().getMedian());
            values.put("min", hist.getValue().getSnapshot().getMin());
            values.put("stdDev", hist.getValue().getSnapshot().getStdDev());
            values.put("75th", hist.getValue().getSnapshot().get75thPercentile());
            values.put("95th", hist.getValue().getSnapshot().get95thPercentile());
            values.put("98th", hist.getValue().getSnapshot().get98thPercentile());
            values.put("99th", hist.getValue().getSnapshot().get99thPercentile());
            values.put("999th", hist.getValue().getSnapshot().get999thPercentile());
            RestcommStatsSender.sendStats(values, "histogram", this.remoteServer);
        }

        //Process Meter Metrics
        for (Map.Entry<String, Meter> meter : meters.entrySet()) {
            values = new HashMap<>();
            values.put("timestamp", timestamp);
            values.put("key", meter.getKey());
            values.put("count", meter.getValue().getCount());
            values.put("fifteenMinuteRate", meter.getValue().getFifteenMinuteRate());
            values.put("fiveMinuteRate", meter.getValue().getFiveMinuteRate());
            values.put("meanRate", meter.getValue().getMeanRate());
            values.put("oneMinuteRate", meter.getValue().getOneMinuteRate());
            RestcommStatsSender.sendStats(values, "meter", this.remoteServer);
        }

        //Process Timer Metrics
        for (Map.Entry<String, Timer> timer : timers.entrySet()) {
            values = new HashMap<>();
            values.put("timestamp", timestamp);
            values.put("key", timer.getKey());
            values.put("count", timer.getValue().getCount());
            values.put("fifteenMinuteRate", timer.getValue().getFifteenMinuteRate());
            values.put("fiveMinuteRate", timer.getValue().getFiveMinuteRate());
            values.put("meanRate", timer.getValue().getMeanRate());
            values.put("oneMinuteRate", timer.getValue().getOneMinuteRate());
            values.put("size", timer.getValue().getSnapshot().size());
            values.put("max", timer.getValue().getSnapshot().getMax());
            values.put("mean", timer.getValue().getSnapshot().getMean());
            values.put("media", timer.getValue().getSnapshot().getMedian());
            values.put("min", timer.getValue().getSnapshot().getMin());
            values.put("stdDev", timer.getValue().getSnapshot().getStdDev());
            values.put("75th", timer.getValue().getSnapshot().get75thPercentile());
            values.put("95th", timer.getValue().getSnapshot().get95thPercentile());
            values.put("98th", timer.getValue().getSnapshot().get98thPercentile());
            values.put("99th", timer.getValue().getSnapshot().get99thPercentile());
            values.put("999th", timer.getValue().getSnapshot().get999thPercentile());
            RestcommStatsSender.sendStats(values, "timer", this.remoteServer);
        }
    }

    public String getRemoteServer() {
        return remoteServer;
    }

    public void setRemoteServer(String remoteServer) {
        this.remoteServer = remoteServer;
    }
    
    /**
     * Static Builder Class
     */
    public static class Builder {

        private final MetricRegistry registry;
        private final Clock clock;
        private final TimeUnit rateUnit;
        private final TimeUnit durationUnit;
        private final MetricFilter filter;

        private Builder(MetricRegistry registry) {
            this.registry = registry;
            this.clock = Clock.defaultClock();
            this.rateUnit = TimeUnit.SECONDS;
            this.durationUnit = TimeUnit.MILLISECONDS;
            this.filter = MetricFilter.ALL;
        }

        public RestcommStatsReporter build() {
            return new RestcommStatsReporter(registry,
                    filter,
                    rateUnit,
                    durationUnit,
                    clock);
        }
    }
}