/*
 * TeleStax, Open Source Cloud Communications
 * Copyright 2011-2014, Telestax Inc and individual contributors
 * by the @authors tag.
 *
 * This program is free software: you can redistribute it and/or modify
 * under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 */

package org.restcomm.media.ice.network.stun;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;

import org.restcomm.media.ice.IceAuthenticator;
import org.restcomm.media.network.deprecated.TransportAddress;
import org.restcomm.media.network.deprecated.TransportAddress.TransportProtocol;
import org.restcomm.media.network.deprecated.channel.PacketHandler;
import org.restcomm.media.network.deprecated.channel.PacketHandlerException;
import org.restcomm.media.stun.StunException;
import org.restcomm.media.stun.messages.StunMessage;
import org.restcomm.media.stun.messages.StunMessageFactory;
import org.restcomm.media.stun.messages.StunRequest;
import org.restcomm.media.stun.messages.StunResponse;
import org.restcomm.media.stun.messages.attributes.StunAttribute;
import org.restcomm.media.stun.messages.attributes.StunAttributeFactory;
import org.restcomm.media.stun.messages.attributes.general.ErrorCodeAttribute;
import org.restcomm.media.stun.messages.attributes.general.MessageIntegrityAttribute;
import org.restcomm.media.stun.messages.attributes.general.PriorityAttribute;
import org.restcomm.media.stun.messages.attributes.general.UsernameAttribute;

/**
 * Handles STUN traffic.
 * 
 * @author Henrique Rosa
 * 
 */
public class StunHandler implements PacketHandler {
	
	private int pipelinePriority;

	private IceAuthenticator iceAuthenticator;
	private final List<StunListener> listeners;
	
	public StunHandler(IceAuthenticator authenticator) {
		this.iceAuthenticator = authenticator;
		this.listeners = new ArrayList<StunListener>();
		this.pipelinePriority = 0;
	}
	
	@Override
	public int getPipelinePriority() {
		return pipelinePriority;
	}
	
	public void setPipelinePriority(int pipelinePriority) {
		this.pipelinePriority = pipelinePriority;
	}
	
	public IceAuthenticator getIceAuthenticator() {
		return iceAuthenticator;
	}
	
	public void setIceAuthenticator(IceAuthenticator iceAuthenticator) {
		this.iceAuthenticator = iceAuthenticator;
	}
	
	public void addListener(StunListener listener) {
		synchronized (this.listeners) {
			if (!this.listeners.contains(listener)) {
				this.listeners.add(listener);
			}
		}
	}
	
	public boolean removeListener(StunListener listener) {
		synchronized (this.listeners) {
			return this.listeners.remove(listener);
		}
	}
	
	private void fireStunBindingEvent(InetSocketAddress localPeer, InetSocketAddress remotePeer) {		
		BindingSuccessEvent event = new BindingSuccessEvent(this, localPeer, remotePeer);
		for (StunListener listener : listeners) {
			listener.onBinding(event);
		}
	}
	
	private long extractPriority(StunRequest request) throws IllegalArgumentException {
		PriorityAttribute priorityAttr = (PriorityAttribute) request.getAttribute(StunAttribute.PRIORITY);
		if (priorityAttr == null) {
			throw new IllegalArgumentException("Missing PRIORITY attribtue!");
		}
		return priorityAttr.getPriority();
	}
	
	private byte[] processRequest(StunRequest request, InetSocketAddress localPeer, InetSocketAddress remotePeer) throws IOException, StunException {
		/*
		 * The agent MUST use a short-term credential to authenticate the
		 * request and perform a message integrity check.
		 */

		byte[] transactionID = request.getTransactionId();

		UsernameAttribute remoteUnameAttribute = (UsernameAttribute) request.getAttribute(StunAttribute.USERNAME);

		// Send binding error response if username is null
		if (remoteUnameAttribute == null) {
			StunResponse errorResponse = new StunResponse();
			errorResponse.setTransactionID(transactionID);
			errorResponse.setMessageType(StunMessage.BINDING_ERROR_RESPONSE);
			errorResponse.addAttribute(StunAttributeFactory.createErrorCodeAttribute(ErrorCodeAttribute.BAD_REQUEST,
					ErrorCodeAttribute.getDefaultReasonPhrase(ErrorCodeAttribute.BAD_REQUEST)));
			return errorResponse.encode();
		}
		String remoteUsername = new String(remoteUnameAttribute.getUsername());
		
		/*
		 * The agent MUST consider the username to be valid if it consists of
		 * two values separated by a colon, where the first value is equal to
		 * the username fragment generated by the agent in an offer or answer
		 * for a session in-progress.
		 */
		if(!this.iceAuthenticator.validateUsername(remoteUsername)) {
			// TODO return error response
			throw new IOException("Invalid username "+ remoteUsername);
		}
		
		/*
		 * The username for the credential is formed by concatenating the
		 * username fragment provided by the peer with the username fragment of
		 * the agent sending the request, separated by a colon (":").
		 */
		int colon = remoteUsername.indexOf(":");
		String localUFrag = remoteUsername.substring(0, colon);
		String remoteUfrag = remoteUsername.substring(colon);
		
		/*
		 * An agent MUST include the PRIORITY attribute in its Binding request.
		 * This priority value will be computed identically to how the priority
		 * for the local candidate of the pair was computed, except that the
		 * type preference is set to the value for peer reflexive candidate
		 * types
		 */
		long priority = extractPriority(request);

		// Produce Binding Response
		TransportAddress transportAddress = new TransportAddress(remotePeer.getAddress(), remotePeer.getPort(), TransportProtocol.UDP);
		StunResponse response = StunMessageFactory.createBindingResponse(request, transportAddress);
		try {
			response.setTransactionID(transactionID);
		} catch (StunException e) {
			throw new IOException("Illegal STUN Transaction ID: " + new String(transactionID), e);
		}

		/*
		 * Add USERNAME and MESSAGE-INTEGRITY attribute in the response. The
		 * responses utilize the same usernames and passwords as the requests
		 */
		String localUsername = remoteUfrag.concat(":").concat(localUFrag);
		StunAttribute unameAttribute = StunAttributeFactory.createUsernameAttribute(localUsername);
		response.addAttribute(unameAttribute);

		byte[] localKey = this.iceAuthenticator.getLocalKey(localUFrag);
		MessageIntegrityAttribute messageIntegrityAttribute = StunAttributeFactory.createMessageIntegrityAttribute(remoteUsername, localKey);
		response.addAttribute(messageIntegrityAttribute);
		
		// If the client issues a USE-CANDIDATE, tell ICE Agent to select the candidate
		if (request.containsAttribute(StunAttribute.USE_CANDIDATE)) {
			fireStunBindingEvent(localPeer, remotePeer);
		}

		// Pass response to the server
		return response.encode();
	}
	
	private byte[] processResponse(StunResponse response) {
		throw new UnsupportedOperationException("Support to handle STUN responses is not implemented.");
	}

	@Override
	public boolean canHandle(byte[] packet) {
		return canHandle(packet, packet.length, 0);
	}

	/*
	 * All STUN messages MUST start with a 20-byte header followed by zero or more Attributes.
	 * The STUN header contains a STUN message type, magic cookie, transaction ID, and message length.
	 * 
     *  0                   1                   2                   3
     *  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
     * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     * |0 0|     STUN Message Type     |         Message Length        |
     * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     * |                         Magic Cookie                          |
     * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     * |                                                               |
     * |                     Transaction ID (96 bits)                  |
     * |                                                               |
     * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     * 
	 * @param data
	 * @param length
	 * @return
	 * @see <a href="http://tools.ietf.org/html/rfc5389#page-10">RFC5389</a>
	 */
	@Override
	public boolean canHandle(byte[] data, int length, int offset) {
		/*
		 * All STUN messages MUST start with a 20-byte header followed by zero
		 * or more Attributes.
		 */
		if(length >= 20) {
			// The most significant 2 bits of every STUN message MUST be zeroes.
			byte b0 = data[offset];
			boolean firstBitsValid = ((b0 & 0xC0) == 0);
			
			// The magic cookie field MUST contain the fixed value 0x2112A442 in network byte order.
			boolean hasMagicCookie = data[offset + 4] == StunMessage.MAGIC_COOKIE[0]
					&& data[offset + 5] == StunMessage.MAGIC_COOKIE[1]
					&& data[offset + 6] == StunMessage.MAGIC_COOKIE[2]
					&& data[offset + 7] == StunMessage.MAGIC_COOKIE[3];
			return firstBitsValid && hasMagicCookie;
		}
		return false;
	}

	@Override
	public byte[] handle(byte[] packet, InetSocketAddress localPeer, InetSocketAddress remotePeer) throws PacketHandlerException {
		return handle(packet, packet.length, 0, localPeer, remotePeer);
	}

	@Override
	public byte[] handle(byte[] packet, int dataLength, int offset, InetSocketAddress localPeer, InetSocketAddress remotePeer) throws PacketHandlerException {
		// Decode and process the packet
		StunMessage message;
		try {
			message = StunMessage.decode(packet, (char) offset, (char) dataLength);
			if (message instanceof StunRequest) {
				return processRequest((StunRequest) message, localPeer, remotePeer);
			} else if (message instanceof StunResponse) {
				return processResponse((StunResponse) message);
			} else {
				// TODO STUN Indication is not supported as of yet
				return null;
			}
		} catch (StunException e) {
			throw new PacketHandlerException("Could not decode STUN packet", e);
		} catch (IOException e) {
			throw new PacketHandlerException(e.getMessage(), e);
		}
	}

	@Override
	public int compareTo(PacketHandler o) {
		if(o == null) {
			return 1;
		}
		return this.getPipelinePriority() - o.getPipelinePriority();
	}

}
