/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.animation.lifecycle;

import java.util.Map;
import java.util.Properties;
import org.robokind.api.animation.messaging.RemoteAnimationPlayerClient;
import org.robokind.api.animation.player.AnimationPlayer;
import org.robokind.api.animation.protocol.AnimationEvent;
import org.robokind.api.animation.protocol.AnimationEvent.AnimationEventFactory;
import org.robokind.api.common.lifecycle.AbstractLifecycleProvider;
import org.robokind.api.common.lifecycle.utils.DescriptorListBuilder;
import org.robokind.api.messaging.Constants;
import org.robokind.api.messaging.MessageSender;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class AnimationPlayerClientLifecycle extends 
        AbstractLifecycleProvider<
                AnimationPlayer, RemoteAnimationPlayerClient> {
    private final static String theAnimationSender = "animSender";
    private final static String theAnimationEventFactory = "animEventFactory";
    private String myPlayerClientId;
    private String myPlayerHostId;
    
    public AnimationPlayerClientLifecycle(
            String animPlayerId, String remotePlayerId, String animSenderId){
        super(new DescriptorListBuilder()
                .dependency(theAnimationSender, MessageSender.class)
                    .with(Constants.PROP_MESSAGE_SENDER_ID, animSenderId)
                    .with(Constants.PROP_MESSAGE_TYPE, 
                            AnimationEvent.class.getName())
                .dependency(theAnimationEventFactory, 
                        AnimationEventFactory.class)
                .getDescriptors());
        if(animPlayerId == null || remotePlayerId == null){
            throw new NullPointerException();
        }
        myPlayerClientId = animPlayerId;
        myPlayerHostId = remotePlayerId;
        myRegistrationProperties = new Properties();
        myRegistrationProperties.put(
                AnimationPlayer.PROP_PLAYER_ID, animPlayerId);
        myServiceClassNames = new String[]{
            AnimationPlayer.class.getName(),
            RemoteAnimationPlayerClient.class.getName()
        };
    }

    @Override
    protected RemoteAnimationPlayerClient create(Map<String, Object> dependencies) {
        MessageSender<AnimationEvent> sender = 
                (MessageSender)dependencies.get(theAnimationSender);
        AnimationEventFactory factory = 
                (AnimationEventFactory)dependencies.get(
                        theAnimationEventFactory);
        RemoteAnimationPlayerClient client = 
                new RemoteAnimationPlayerClient(
                        myPlayerClientId, myPlayerHostId);
        client.setAnimationEventFactory(factory);
        client.setAnimationEventSender(sender);
        return client;
    }

    @Override
    protected void handleChange(String name, Object dependency, 
            Map<String,Object> availableDependencies){
        if(myService == null){
            return;
        }
        if(theAnimationSender.equals(name)){
            myService.setAnimationEventSender(
                    (MessageSender)dependency);
        }else if(theAnimationEventFactory.equals(name)){
            myService.setAnimationEventFactory(
                    (AnimationEventFactory)dependency);
        }
    }

    @Override
    public Class<AnimationPlayer> getServiceClass() {
        return AnimationPlayer.class;
    }
}
