/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.animation.messaging;

import java.util.Collections;
import java.util.List;
import org.robokind.api.animation.Animation;
import org.robokind.api.animation.Channel;
import org.robokind.api.animation.MotionPath;
import org.robokind.api.animation.player.AnimationJob;
import org.robokind.api.animation.player.AnimationPlayer;
import org.robokind.api.animation.protocol.AnimationEvent;
import org.robokind.api.animation.protocol.AnimationEvent.AnimationEventFactory;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.api.common.playable.Playable;
import org.robokind.api.common.services.addon.ServiceAddOn;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.messaging.MessageSender;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RemoteAnimationPlayerClient implements AnimationPlayer{
    private String myAnimationPlayerId;
    private String myRemotePlayerId;
    private MessageSender<AnimationEvent> myAnimationSender;
    private AnimationEventFactory myEventFactory;
    private List<RemoteAnimationJob> myAnimationJobs;

    public RemoteAnimationPlayerClient(
            String animPlayerClientId, String animPlayerHostId){
        if(animPlayerClientId == null || animPlayerHostId == null){
            throw new NullPointerException();
        }
        myAnimationPlayerId = animPlayerClientId;
        myRemotePlayerId = animPlayerHostId;
    }
    
    @Override
    public String getAnimationPlayerId(){
        return myAnimationPlayerId;
    }
    
    public void setAnimationEventFactory(AnimationEventFactory factory){
        myEventFactory = factory;
    }
    
    public void setAnimationEventSender(MessageSender<AnimationEvent> sender){
        myAnimationSender = sender;
    }
    
    @Override
    public AnimationJob playAnimation(Animation animation) {
        if(myEventFactory == null || myAnimationSender == null){
            return null;
        }
        AnimationEvent event = 
                myEventFactory.createAnimationEvent(
                        myAnimationPlayerId, myRemotePlayerId, animation);
        myAnimationSender.notifyListeners(event);
        for(ServiceAddOn<Playable> add : animation.getAddOns()){
            add.getAddOn().start(TimeUtils.now());
        }
        AnimationJob job = new RemoteAnimationJob(this, animation,
                animation.getStartTime(), animation.getStopTime(),  800);
        job.start(TimeUtils.now());
        return job;
    }
    public AnimationJob loopAnimation(Animation animation) {
        if(myEventFactory == null || myAnimationSender == null){
            return null;
        }
        AnimationEvent event = 
                myEventFactory.createAnimationEvent(
                        myAnimationPlayerId, "LOOP", animation);
        myAnimationSender.notifyListeners(event);
        for(ServiceAddOn<Playable> add : animation.getAddOns()){
            add.getAddOn().start(TimeUtils.now());
        }
        AnimationJob job = new RemoteAnimationJob(this, animation, 
                animation.getStartTime(), animation.getStopTime(),  800);
        job.start(TimeUtils.now());
        return job;
    }
    
    public void stopAnimation(Animation animation) {
        if(myEventFactory == null || myAnimationSender == null){
            return;
        }
        AnimationEvent event = 
                myEventFactory.createAnimationEvent(
                        myAnimationPlayerId, "STOP", animation);
        myAnimationSender.notifyListeners(event);
        for(ServiceAddOn<Playable> add : animation.getAddOns()){
            add.getAddOn().stop(TimeUtils.now());
        }
    }
    
    public void clearAnimations() {
        if(myEventFactory == null || myAnimationSender == null){
            return;
        }
        Animation empty = new Animation(new VersionProperty("empty", "1.0"));
        empty.addChannel(new Channel(0, "emptyChan"));
        MotionPath path = new MotionPath();
        path.addPoint(0, 0.5);
        path.addPoint(1, 0.5);
        empty.getChannel(0).addPath(path);
        AnimationEvent event = 
                myEventFactory.createAnimationEvent(
                        myAnimationPlayerId, "CLEAR", empty);
        myAnimationSender.notifyListeners(event);
        return;
    }

    @Override
    public AnimationJob playAnimation(Animation animation, Long start, Long stop) {
        animation.setStartTime(start);
        animation.setStopTime(stop);
        return playAnimation(animation);
    }

    @Override
    public List<AnimationJob> getCurrentAnimations() {
        return (List)myAnimationJobs;
    }

    @Override
    public void removeAnimationJob(AnimationJob job) {
        if(!(job instanceof RemoteAnimationJob)){
            return;
        }
        RemoteAnimationJob rjob = (RemoteAnimationJob)job;
        if(!myAnimationJobs.contains(rjob)){
            return;
        }
        rjob.stop(TimeUtils.now());
        myAnimationJobs.remove(rjob);
    }
    
}
