/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.api.common.osgi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public abstract class ServiceClassListener<T> implements ServiceListener{
    private final static Logger theLogger = Logger.getLogger(ServiceClassListener.class.getName());
    protected List<T> myList;
    private Class<T> myClass;
    protected BundleContext myContext;
    private Map<ServiceReference,T> myReferenceMap;

    /**
     *
     * @param context
     * @param list
     */
    public ServiceClassListener(Class<T> clazz, BundleContext context, String serviceFilter){
        myContext = context;
        myList = new ArrayList<T>();
        myReferenceMap = new HashMap();
        myClass = clazz;
        if(myContext != null){
            String filter = "(" + Constants.OBJECTCLASS + "=" + myClass.getName() + ")";
            boolean empty = (serviceFilter == null || serviceFilter.isEmpty());
            filter = empty ? filter : "(&" + filter + serviceFilter + ")";
            try{
                myContext.addServiceListener(this, filter);
            }catch(InvalidSyntaxException ex){
                theLogger.log(Level.WARNING, "Could not register ServiceListener.  Invalid filter syntax.", ex);
            }
        }
        if(myList == null || myContext == null){
            return;
        }
        try{
            ServiceReference[] refs = context.getServiceReferences(myClass.getName(),null);
            if(refs == null || refs.length == 0){
                return;
            }
            for(ServiceReference se : refs){
                addService(se);
            }
        }catch(InvalidSyntaxException ex){
            theLogger.log(Level.SEVERE, "There was an error fetching service references.", ex);
        }
    }

    @Override
    public void serviceChanged(ServiceEvent se) {
        ServiceReference ref = se.getServiceReference();
        switch(se.getType()){
            case ServiceEvent.REGISTERED: addService(ref); break;
            case ServiceEvent.MODIFIED_ENDMATCH:
            case ServiceEvent.UNREGISTERING: removeService(ref); break;
        }
    }

    private void addService(ServiceReference ref){
        T t = getService(ref);
        if(t == null){
            return;
        }
        if(!myReferenceMap.containsKey(ref)){
            myList.add(t);
            myReferenceMap.put(ref,t);
            addService(t);
        }
    }
    
    protected abstract void addService(T t);

    private void removeService(ServiceReference ref){
        T t = getService(ref);
        myReferenceMap.remove(ref);
        myList.remove(t);
        removeService(t);
    }
    
    protected abstract void removeService(T t);

    private T getService(ServiceReference se) {
        if (se == null || myContext == null) {
            return null;
        }
        Object obj = myContext.getService(se);
        if (!myClass.isInstance(obj)) {
            return null;
        }
        T t = (T) obj;
        return t;
    }
}
