/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.osgi.lifecycle;

import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Abstract implementation of ServiceLifecycleProvider.  Automatically validates
 * dependencies against the descriptors provided at creation.
 * 
 * @param <I> Interface or Base Class of the service managed by this lifecycle
 * @param <T> actual type of the service managed
 * @author Matthew Stevenson <www.robokind.org>
 */
public abstract class AbstractLifecycleProvider<I,T extends I> implements 
        ServiceLifecycleProvider<I> {

    private List<DependencyDescriptor> myDescriptors;
    /**
     * Instance of the service being managed
     */
    protected T myService;
    /**
     * OSGi registration properties
     */
    protected Properties myRegistrationProperties;
    
    /**
     * Creates a new AbstractLifecycleProvider with the given 
     * DependencyDescriptors.
     * @param deps descriptors of the dependencies for this lifecycle
     */
    public AbstractLifecycleProvider(List<DependencyDescriptor> deps){
        if(deps == null){
            throw new NullPointerException();
        }
        myDescriptors = deps;
    }

    @Override
    public void start(Map<String, Object> dependencies) {
        if(dependencies == null){
            throw new NullPointerException();
        }
        if(!Validator.validateServices(myDescriptors, dependencies)){
            throw new IllegalArgumentException(
                    "Invalid Dependency set for factory.");
        }
        myService = create(dependencies);
    }
    /**
     * Called when all dependencies are available.  This should create and 
     * return the service for this lifecycle.
     * @param dependencies validated map of the lifecycle's dependencies
     * @return the service managed by this lifecycle, null if the service
     * cannot be created
     */
    protected abstract T create(Map<String,Object> dependencies);

    @Override
    public T getService() {
        return myService;
    }
    
    @Override
    public Properties getRegistrationProperties(){
        return myRegistrationProperties;
    }

    @Override
    public List<DependencyDescriptor> getDependencyDescriptors() {
        return myDescriptors;
    }

    @Override
    public void dependencyChanged(String serviceId, Object service) {
        if(serviceId == null){
            throw new NullPointerException();
        }
        validate(serviceId, service);
        handleChange(serviceId, service);
    }
    /**
     * Called from <code>dependencyChanged</code> with validated values.
     * @param dependencyId id of the dependency changed 
     * @param service new dependency, null if the dependency was removed
     */
    protected abstract void handleChange(String dependencyId, Object service);
    
    private void validate(String id, Object req){
        List<DependencyDescriptor> reqs = getDependencyDescriptors();
        if(req != null){
            if(Validator.validateService(reqs, id, req)){
                return;
            }
            throw new IllegalArgumentException("Invalid service or id.  "
                    + "id: " + id + ", service: " + req);
        }
        if(Validator.validateServiceId(reqs, id)){
            return;
        }
        throw new IllegalArgumentException("Invalid service id: " + id + ".");
    }
    
}
