/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.osgi.lifecycle;

import org.robokind.api.common.osgi.OSGiUtils;

/**
 * Describes a service dependency of a ServiceLifecycleProvider.  Used to match
 * find and track the appropriate services needed by a lifecycle provider.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DependencyDescriptor {
    private String myDependencyId;
    private Class myDependencyClass;
    private String myDependencyFilter;
    
    /**
     * Builds a descriptor for a service with a given id.
     * Creates a new DependencyDescriptor with a filter for a given serviceId 
     * using the given idPropertyName. Robot.PROP_ID is an example of a 
     * property name, and the Robot's id would be the serviceId.
     * @param localId string used to identify this dependency within a
     * ServiceLifecycleProvider
     * @param clazz class of the dependency
     * @param idPropertyName property name for the service's id
     * @param serviceId the service id the dependency must match
     * @param filter optional filter string the dependency must match.  This may
     * be left null.
     * @return new DependencyDescriptor for a service with a given id
     */
    public static DependencyDescriptor buildForServiceId(
            String localId, Class clazz, 
            String idPropertyName, String serviceId, String filter){
        String idFilter = 
                OSGiUtils.createIdFilter(idPropertyName, serviceId, filter);
        return new DependencyDescriptor(
                localId, clazz, idFilter);
    }
    
    /**
     * Create a new DependencyDescriptor with the given values.
     * @param dependencyId local id used within a ServiceLifecycleProvider
     * @param clazz Class of the dependency
     * @param filter optional OSGi filter string
     */
    public DependencyDescriptor(String dependencyId, 
            Class clazz, String filter){
        if(dependencyId == null || clazz == null){
            throw new NullPointerException();
        }
        myDependencyId = dependencyId;
        myDependencyClass = clazz;
        myDependencyFilter = filter;
    }

    /**
     * Returns the local id String used within a ServiceLifecycleProvider.
     * @return local id String used within a ServiceLifecycleProvider
     */
    public String getDependencyId(){
        return myDependencyId;
    }
    /**
     * Returns the Class of dependency.
     * @return Class of dependency
     */
    public Class getServiceClass(){
        return myDependencyClass;
    }

    /**
     * Returns an OSGi filter string for the dependency, null if it is
     * not set.
     * @return OSGi filter string for the dependency, null if it is not set
     */
    public String getServiceFilter(){
        return myDependencyFilter;
    }
}
