/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.osgi.lifecycle;

import java.util.ArrayList;
import java.util.List;

/**
 * Convenience class for building a List of DependencyDescriptors.
 * Methods for adding descriptors return the DescriptorListBuilder to allow for
 * method chaining.
 * e.x.:
 *      List&gt;DependencyDescriptor&lt; descs = 
 *          new DescriptorListBuilder().add(...).addId(...).getDescriptors();
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DescriptorListBuilder{
    private List<DependencyDescriptor> myDependencies;
    /**
     * Creates an empty DescriptorListBuilder.
     */
    public DescriptorListBuilder(){
        myDependencies = new ArrayList<DependencyDescriptor>();
    }
    /**
     * Creates and adds a DependencyDescriptor with the given values.
     * @param localDependencyId dependency id used with a 
     * ServiceLifecycleProvider
     * @param clazz dependency class
     * @param idPropertyName property name for the dependency's service id
     * @param serviceId service id of the dependency
     * @return this list builder
     */
    public DescriptorListBuilder addId(
            String localDependencyId, Class clazz, 
            String idPropertyName, String serviceId){
        return addId(localDependencyId, clazz, 
                idPropertyName, serviceId, null);
    }
    /**
     * Creates and adds a DependencyDescriptor with the given values.
     * @param localDependencyId dependency id used with a 
     * ServiceLifecycleProvider
     * @param clazz dependency class
     * @param idPropertyName property name for the dependency's service id
     * @param serviceId service id of the dependency
     * @param filter optional OSGi filter String
     * @return this list builder
     */
    public DescriptorListBuilder addId(
            String localDependencyId, Class clazz, 
            String idPropertyName, String serviceId, String filter){
        myDependencies.add(
                DependencyDescriptor.buildForServiceId(
                        localDependencyId, clazz, 
                        idPropertyName, serviceId, filter));
        return this;
    }
    /**
     * 
     * Creates and adds a DependencyDescriptor with the given values.
     * @param localDependencyId dependency id used with a 
     * ServiceLifecycleProvider
     * @param clazz dependency class
     * @param filter optional OSGi filter String
     * @return this list builder
     */
    public DescriptorListBuilder add(
            String localDependencyId, Class clazz, String filter){
        myDependencies.add(
                new DependencyDescriptor(
                        localDependencyId, clazz, filter));
        return this;
    }
    /**
     * Returns the list of DependencyDescriptors added to this list builder.
     * @return list of DependencyDescriptors added to this list builder
     */
    public List<DependencyDescriptor> getDescriptors(){
        return myDependencies;
    }
}
