/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.osgi.lifecycle;

import org.robokind.api.common.osgi.lifecycle.ServiceLifecycleProvider.Validator;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import static org.robokind.api.common.osgi.lifecycle.ServiceDependencyTracker.*;

/**
 * Launches and manages a ServiceLifecycleProvider.  Creates a 
 * ServiceDependencyTracker to drive the service lifecycle from the OSGi
 * Service Registry.  The the service lifecycle changes, the 
 * DynamicServiceLauncher registers/unregisters the managed service in the
 * OSGi ServiceRegistry.
 * 
 * @param <T> type of the dynamic service to be launched
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DynamicServiceLauncher<T>{
    private final static Logger theLogger = 
            Logger.getLogger(DynamicServiceLauncher.class.getName());
    private BundleContext myContext;
    private ServiceLifecycleProvider<T> myLifecycleProvider;
    private ServiceDependencyTracker myRequirementsTracker;
    private T myService;
    private ServiceRegistration myServiceRegistration;
    private Properties myRegistrationProperties;
    private String myRegistrationClassName;
    private boolean myInitializedFlag;
    /**
     * Creates a new DynamicServiceLauncher from the given lifecycle provider.
     * @param context BundleContext for accessing the OSGi Service Registry
     * @param lifecycle lifecycle provider for the managed service 
     */
    public DynamicServiceLauncher(
            BundleContext context, ServiceLifecycleProvider<T> lifecycle){
        this(context, lifecycle, lifecycle.getServiceClass().getName(), null);
    }
    /**
     * Creates a new DynamicServiceLauncher from the given lifecycle provider.
     * Uses the given registration properties when registering the dynamic
     * service to the OSGi Service Registry.
     * @param context BundleContext for accessing the OSGi Service Registry
     * @param lifecycle lifecycle provider for the dynamic service 
     * @param registrationProps optional properties to be used when registering 
     * the dynamic service to the OSGi Service Registry.  These are combined
     * with any properties from the ServiceLifecycleProvider
     */
    public DynamicServiceLauncher(
            BundleContext context, ServiceLifecycleProvider<T> lifecycle, 
            Properties registrationProps){
        this(context, lifecycle, 
                lifecycle.getServiceClass().getName(), registrationProps);
    }
    /**
     * Creates a new DynamicServiceLauncher from the given lifecycle provider.
     * Uses the given registration class name and properties when registering 
     * the dynamic service to the OSGi Service Registry.
     * @param context BundleContext for accessing the OSGi Service Registry
     * @param lifecycle lifecycle provider for the dynamic service 
     * @param registrationProps optional properties to be used when registering 
     * the dynamic service to the OSGi Service Registry.  These are combined
     * with any properties from the ServiceLifecycleProvider
     * @param registrationClassName class name to be used when registering the
     * dynamic service to the OSGi Service Registry.  By default, this value is
     * taken from the ServiceLifecycleProvider's getServiceClass().
     */
    public DynamicServiceLauncher(
            BundleContext context, ServiceLifecycleProvider<T> lifecycle, 
            String registrationClassName, Properties registrationProps){
        if(context == null || lifecycle == null 
                || registrationClassName == null){
            throw new NullPointerException();
        }
        myContext = context;
        myLifecycleProvider = lifecycle;
        List<DependencyDescriptor> descriptors = 
                myLifecycleProvider.getDependencyDescriptors();
        if(descriptors == null){
            throw new NullPointerException();
        }
        myRequirementsTracker = new ServiceDependencyTracker(myContext);
        for(DependencyDescriptor dd : descriptors){
            myRequirementsTracker.addDependencyDescription(
                    dd.getServiceClass(), 
                    dd.getDependencyId(), 
                    dd.getServiceFilter());
        }
        myRequirementsTracker.addPropertyChangeListener(
                new DependencyStatusListener());
        myRegistrationClassName = registrationClassName;
        myRegistrationProperties = registrationProps;
        myInitializedFlag = false;
    }
    /**
     * Starts the DynamicServiceLauncher.  Begins tracker dependencies and
     * notifies the lifecycle provider of dependency changes.
     */
    public void start(){
        myRequirementsTracker.start();
    }
    /**
     * Stops tracking dependency changes.  Does not modify the existing service
     * or its OSGi registration state.
     */
    public void stop(){
        myRequirementsTracker.stop();
    }
    /**
     * Unregisters the dynamic service from the OSGi Service Registry.
     */
    public void unregisterService(){
        myService = null;
        if(myServiceRegistration == null){
            return;
        }
        myServiceRegistration.unregister();
        myServiceRegistration = null;
    }
    
    private void handleAllDependencies(Map<String,Object> requiredServices){
        if(myService != null){
            return;
        }else if(!ServiceLifecycleProvider.Validator.validateServices(
                myLifecycleProvider.getDependencyDescriptors(), 
                requiredServices)){
            throw new IllegalArgumentException(
                    "Invalid dependency set for service.");
        }
        myLifecycleProvider.start(requiredServices);
        myService = myLifecycleProvider.getService();
        if(myService == null){
            theLogger.warning("The factory failed to create a service "
                    + "and returned null.");
            return;
        }
        theLogger.log(Level.INFO, 
                "Service created of type {0}", 
                myLifecycleProvider.getServiceClass());
        registerService();
    }
    
    private void handleChanged(String id, Object newDependency){
        if(id == null){
            throw new NullPointerException();
        }else if(!validate(id, newDependency)){
            throw new IllegalArgumentException("Invalid id or dependency.  "
                    + "id: " + id + ", dependency: " + newDependency);
        }
        myLifecycleProvider.dependencyChanged(id, newDependency);
        checkForModification();
    }
    
    private boolean validate(String id, Object req){
        List<DependencyDescriptor> reqs = 
                myLifecycleProvider.getDependencyDescriptors();
        return (req == null && Validator.validateServiceId(reqs, id)) ||
                Validator.validateService(reqs, id, req);
    }
    
    private void checkForModification(){
        T service = myLifecycleProvider.getService();
        if(service == null && myService != null){
            theLogger.warning("Required Service change stopped this service.  "
                    + "Service is being unregistered.");
            unregisterService();
        }else if(service != null && myService == null){
            theLogger.info(
                    "Required Service change has started this service.");
            myService = service;
            registerService();
        }else if(service != myService){
            theLogger.warning(
                    "Required Service change has changed this service.");
            ServiceRegistration oldReg = myServiceRegistration;
            myService = service;
            registerService();
            oldReg.unregister();
        }
    }
    
    /**
     * Registers the dynamic service to the OSGi Service Registry.
     */
    protected void registerService(){
        Properties props = myLifecycleProvider.getRegistrationProperties();
        Properties allProps = null;
        if(props != null || myRegistrationProperties != null){
            allProps = new Properties();
            if(props != null){
                allProps.putAll(props);
            }
            if(myRegistrationProperties != null){
                allProps.putAll(myRegistrationProperties);
            }
        }
        ServiceRegistration reg = myContext.registerService(
                myRegistrationClassName, myService, allProps);
        myServiceRegistration = reg;
        theLogger.log(Level.INFO, 
                "Service Successfully Registered: {0}", myService.toString());
    }
    
    class DependencyStatusListener implements PropertyChangeListener{

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if(evt == null){
                return;
            }
            String name = evt.getPropertyName();
            if(PROP_ALL_DEPENDENCIES_AVAILABLE.equals(name)){
                handleAllAvailable(evt);
            }else if(!myInitializedFlag){
                return;
            }else if(PROP_DEPENDENCY_CHANGED.equals(name)){
                String id = evt.getOldValue().toString();
                Object obj = evt.getNewValue();
                handleChanged(id, obj);
            }
        }
        
        private boolean handleAllAvailable(PropertyChangeEvent evt){
            theLogger.info("All requirements available, "
                    + "attempting to create service.");
            Object obj = evt.getNewValue();
            if(obj == null || !(obj instanceof Map)){
                theLogger.warning(
                        "Invalid requirement map, cannot create service.");
                return true;
            }try{
                Map<String,Object> reqs = (Map<String,Object>)obj;
                handleAllDependencies(reqs);
            }catch(ClassCastException ex){
                theLogger.log(Level.WARNING, 
                        "Improper requirement Map type.", ex);
                return true;
            }
            myInitializedFlag = true;
            return true;
        }
    }
}
