/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.osgi.lifecycle;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.robokind.api.common.osgi.SingleServiceListener;
import org.robokind.api.common.property.PropertyChangeNotifier;

/**
 * Monitors the OSGi Service Registry for a set of service dependencies.  
 * Fires property change events when all of the dependencies are available, and 
 * as the dependencies change.
 * 
 * Used with DynamicServiceLauncher and ServiceLifecycleProvider to create 
 * services with OSGi-driven lifecycles.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class ServiceDependencyTracker extends PropertyChangeNotifier{
    private final static Logger theLogger = 
            Logger.getLogger(ServiceDependencyTracker.class.getName());
    /**
     * Property change event name for a dependency becoming available.
     */
    public final static String PROP_DEPENDENCY_AVAILABLE = "dependencyAvailable";
    /**
     * Property change event name for a dependency changing.
     */
    public final static String PROP_DEPENDENCY_CHANGED = "dependencyChanged";
    /**
     * Property change event name for a dependency becoming unavailable.
     */
    public final static String PROP_DEPENDENCY_UNAVAILABLE = "dependencyUnavailable";
    /**
     * Property change event name for all dependencies being available.
     */
    public final static String PROP_ALL_DEPENDENCIES_AVAILABLE = "allDependenciesAvailable";
    
    private BundleContext myContext;
    private List<SingleServiceListener> myDependencyTrackers;
    private Map<String,SingleServiceListener> myDependencyIdMap;
    private Map<String,Object> myAvailableRequirements;
    private boolean myListeningFlag;
    
    /**
     * Creates an empty ServiceDependencyTracker with the given BundleContext.
     * @param context 
     */
    public ServiceDependencyTracker(BundleContext context){
        if(context == null){
            throw new NullPointerException();
        }
        myContext = context;
        myDependencyTrackers = new ArrayList<SingleServiceListener>();
        myAvailableRequirements = new HashMap();
        myDependencyIdMap = new HashMap<String, SingleServiceListener>();
        myListeningFlag = false;
    }
    
    /**
     * Returns true if all dependencies are available.
     * @return true if all dependencies are available
     */
    public boolean dependenciesSatisfied(){
        return myAvailableRequirements.size() == myDependencyTrackers.size();
    }
    /**
     * Returns a map of dependency Ids dependencies.  Returns null unless all
     * dependencies are available.
     * @return map of dependency Ids dependencies, null unless all
     * dependencies are available
     */
    public Map<String,Object> getAvailableDependencies(){
        if(!dependenciesSatisfied()){
            return null;
        }
        return myAvailableRequirements;
    }
    
    /**
     * Returns the dependency matching the given id, null if unavailable.
     * @param dependencyId local id used with a ServiceLifecycleProvider
     * @return service matching the DependencyDescriptor with the given id, null
     * if the dependency is unavailable
     */
    public Object getDependency(String dependencyId){
        return myAvailableRequirements.get(dependencyId);
    }
    /**
     * Adds the description to the list of dependency to listen for.
     * @param descriptor dependency description to listen for
     * @throws IllegalStateException if the tracker has already been started
     * @throws IllegalArgumentException if the given dependencyId already exists
     */
    public void addDependencyDescription(DependencyDescriptor descriptor){
        if(descriptor  == null){
            throw new NullPointerException();
        }
        addDependencyDescription(
                descriptor.getServiceClass(), 
                descriptor.getDependencyId(), 
                descriptor.getServiceFilter());
    }
    /**
     * Adds the description to the list of dependency to listen for.
     * Descriptions cannot be added once the tracker has been started.
     * @param clazz dependency class
     * @param dependencyId local dependency id to be used with a 
     * ServiceLifecycleProvider
     * @param filterString optional OSGi filter String for the dependency
     * @throws IllegalStateException if the tracker has already been started
     * @throws IllegalArgumentException if the given dependencyId already exists
     */
    public void addDependencyDescription(
            Class clazz, String dependencyId, String filterString){
        if(clazz == null || dependencyId == null){
            throw new NullPointerException();
        }
        if(myListeningFlag){
            throw new IllegalStateException(
                    "Unable to add requirement while listening.");
        }else if(myDependencyIdMap.containsKey(dependencyId)){
            throw new IllegalArgumentException(
                    "Unable to add requirement, requirementId already exists.");
        }
        SingleServiceListener ssl = 
                new SingleServiceListener(clazz, myContext, filterString);
        ssl.addPropertyChangeListener(new RequirementListener(dependencyId));
        myDependencyTrackers.add(ssl);
        myDependencyIdMap.put(dependencyId, ssl);
    }
    /**
     * Start tracking dependencies.
     */
    public void start(){
        myListeningFlag = true;
        for(SingleServiceListener ssl : myDependencyTrackers){
            ssl.start();
        }
    }
    /**
     * Stop tracking dependencies.
     */
    public void stop(){
        myListeningFlag = false;
        myAvailableRequirements.clear();
        for(SingleServiceListener ssl : myDependencyTrackers){
            ssl.stop();
        }
    }
    
    private void requiredServiceFound(String requirementId, Object req){
        if(requirementId == null || req == null){
            throw new NullPointerException();
        }
        if(!myDependencyIdMap.containsKey(requirementId)){
            return;
        }
        theLogger.log(Level.INFO, "Found required service: {0}", requirementId);
        myAvailableRequirements.put(requirementId, req);
        firePropertyChange(PROP_DEPENDENCY_AVAILABLE, requirementId, req);
        checkAllRequirements();
    }
    
    private void checkAllRequirements(){
        if(!dependenciesSatisfied()){
            return;
        }
        theLogger.log(Level.INFO, "All requirements present: {0}", 
                Arrays.toString(myDependencyIdMap.keySet().toArray()));
        firePropertyChange(PROP_ALL_DEPENDENCIES_AVAILABLE, null, getAvailableDependencies());
        
    }
    
    private void requiredServiceChanged(String requirementId, Object req){
        if(requirementId == null || req == null){
            throw new NullPointerException();
        }
        if(!myDependencyIdMap.containsKey(requirementId)){
            return;
        }
        theLogger.log(Level.INFO, "Required service changed: {0}", requirementId);
        myAvailableRequirements.put(requirementId, req);
        firePropertyChange(PROP_DEPENDENCY_CHANGED, requirementId, req);
    }
    
    private void requiredServiceLost(String requirementId){
        if(requirementId == null){
            throw new NullPointerException();
        }
        if(!myDependencyIdMap.containsKey(requirementId)){
            return;
        }
        theLogger.log(Level.INFO, "Lost required service: {0}", requirementId);
        myAvailableRequirements.remove(requirementId);
        firePropertyChange(PROP_DEPENDENCY_UNAVAILABLE, requirementId, null);
    }
    
    class RequirementListener implements PropertyChangeListener {
        private String myRequirementId;
        
        public RequirementListener(String requirementId){
            myRequirementId = requirementId;
        }
        
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if(SingleServiceListener.PROP_SERVICE_TRACKED.equals(evt.getPropertyName())){
                track(evt.getOldValue(), evt.getNewValue());
            }else if(SingleServiceListener.PROP_SERVICE_REMOVED.equals(evt.getPropertyName())){
                untrack(evt.getNewValue());
            }
        }
        
        private void track(Object oldVal, Object newVal){
            if(oldVal == null){
                requiredServiceFound(myRequirementId, newVal);
            }else{
                requiredServiceChanged(myRequirementId, newVal);
            }
        }
        
        private void untrack(Object obj){
            requiredServiceLost(myRequirementId);
        }
    }
}
