/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.api.common.services;

import org.robokind.api.common.config.VersionProperty;

/**
 * An ServiceContext keeps track of an attempt to create a new Service.  It 
 * contains a complementary pair of ServiceConfigurationLoader and 
 * ServiceFactory where: 
 * the loader takes the given Param type
 * the loader and connector share the same ServiceConfig class
 * the connector returns the given Service class
 * @param <ServiceClass> Class of Service created in this ServiceContext
 * @param <ServiceConfig> Class of ServiceConfig used in this ServiceContext 
 * @param <Param> Class of the parameter used by this ServiceContext
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class ServiceContext<ServiceClass,ServiceConfig,Param> {
    private ServiceConfigurationLoader<ServiceConfig, Param> myLoader;
    private ServiceFactory<ServiceClass, ServiceConfig> myFactory;
    private Param myLoadParameter;
    private ServiceConfig myConfig;
    private ServiceClass myService;
    /**
     * Creates an empty ServiceContext
     */
    public ServiceContext(){}
    /**
     * Creates a new ServiceContext from the given 
     * ServiceConfigurationLoader and ServiceFactory.
     * @param loader ServiceConfigurationLoader of the correct type
     * @param connector ServiceFactory of the correct type
     */
    public ServiceContext(
            ServiceConfigurationLoader<ServiceConfig, Param> loader, 
            ServiceFactory<ServiceClass, ServiceConfig> connector){
        if(loader == null || connector == null){
            throw new NullPointerException();
        }
        myLoader = loader;
        myFactory = connector;
    }    
    /**
     * Returns the ServiceConfigurationLoader for this ServiceContext.
     * @return ServiceConfigurationLoader for this ServiceContext
     */
    public ServiceConfigurationLoader<ServiceConfig, Param> getServiceConfigurationLoader(){
        return myLoader;
    }
    /**
     * Sets the ServiceConfigurationLoader for this ServiceContext.
     * @param loader ServiceConfigurationLoader to use
     */
    public void setServiceConfigurationLoader(ServiceConfigurationLoader<ServiceConfig, Param> loader){
        myLoader = loader;
    }
    /**
     * Return the ServiceFactory used by this ServiceContext.
     * @return ServiceFactory used by this ServiceContext
     */
    public ServiceFactory<ServiceClass, ServiceConfig> getServiceFactory(){
        return myFactory;
    }
    /**
     * Sets the ServiceFactory to be used by this ServiceContext.
     * @param factory ServiceFactory to be used by this ServiceContext
     */
    public void setServiceConnector(ServiceFactory<ServiceClass, ServiceConfig> factory){
        myFactory = factory;
    }
    /**
     * Returns the load parameter.
     * @return load parameter
     */
    public Param getLoadParameter(){
        return myLoadParameter;
    }
    /**
     * Sets the load parameter.  This is passed to the 
     * ServiceConfigurationLoader to load the ServiceConfig.
     * @param param parameter passed to the ServiceConfigurationLoader to load 
     * the ServiceConfig
     */
    public void setLoadParameter(Param param){
        myLoadParameter = param;
    }
    /**
     * Loads the ServiceConfig using the ServiceConfigurationLoader and 
     * LoadParameter.
     */
    public boolean loadConfiguration() throws Exception{
        if(myConfig != null){
            return true;
        }
        if(myLoader == null){
            throw new NullPointerException();
        }
        myConfig = myLoader.loadConfiguration(myLoadParameter);
        return myConfig != null;
    }
    /**
     * Returns the ServiceConfig loaded from the LoadParameter. 
     * loadConfiguration must be called for the configuration to be available.
     * @return ServiceConfig loaded from the LoadParameter
     */
    public ServiceConfig getServiceConfiguration(){
        return myConfig;
    }
    /**
     * Builds a Service using the ServiceFactory and ServiceConfig.  Attempts
     * to load the ServiceConfig if it has not been loaded.
     */
    public boolean buildService() throws Exception{
        if(myService != null){
            return true;
        }
        if(myFactory == null){
            return false;
        }
        if(myConfig == null){
            loadConfiguration();
            if(myConfig == null){
                return false;
            }
        }
        myService = myFactory.build(myConfig);
        return myService != null;
    }
    /**
     * Returns the Service from this context.  buildService() must be called
     * fore the service to be available.
     * @return Service from this context
     */
    public ServiceClass getService(){
        return myService;
    }
    
    public VersionProperty getServiceVersion(){
        return myFactory.getServiceVersion();
    }
    
    public Class<ServiceClass> getServiceClass(){
        return myFactory.getServiceClass();
    }
    
    public VersionProperty getConfigFormat(){
        return myLoader.getConfigurationFormat();
    }
    
    public Class<ServiceConfig> getConfigClass(){
        return myLoader.getConfigurationClass();
    }
}
