/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.services;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.api.common.osgi.OSGiUtils;
import org.robokind.api.common.osgi.SingleServiceListener;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class ServiceUtils {
    private final static Logger theLogger = 
            Logger.getLogger(ServiceUtils.class.getName());
    
    public static ServiceRegistration registerServiceFactory(
            BundleContext context, ServiceFactory fact, Properties props){
        if(context == null || fact == null){
            throw new NullPointerException();
        }
        if(props == null){
            props = new Properties();
        }
        props.put(ServiceFactory.PROP_SERVICE_VERSION, 
                fact.getServiceVersion().toString());
        props.put(ServiceFactory.PROP_SERVICE_CLASS, 
                fact.getServiceClass().getName());
        ServiceRegistration reg = context.registerService(
                ServiceFactory.class.getName(), fact, props);
        if(reg == null){
            theLogger.log(Level.WARNING, 
                    "Unknown error occured while registering ServiceFactory.");
        }else{
            theLogger.log(Level.INFO, 
                    "ServiceFactory registered successfully.");
        }
        return reg;
    }
    
    public static ServiceRegistration registerConfigurationLoader(
            BundleContext context, ServiceConfigurationLoader loader, 
            Properties props){
        if(context == null || loader == null){
            throw new NullPointerException();
        }
        if(props == null){
            props = new Properties();
        }
        props.put(ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION, 
                loader.getConfigurationFormat().toString());
        props.put(ServiceConfigurationLoader.PROP_SERVICE_CONFIG_CLASS, 
                loader.getConfigurationClass().getName());
        props.put(ServiceConfigurationLoader.PROP_PARAMETER_CLASS, 
                loader.getParameterClass().getName());
        ServiceRegistration reg = context.registerService(
                ServiceConfigurationLoader.class.getName(), loader, props);
        if(reg == null){
            theLogger.log(Level.WARNING, 
                    "Unknown error occured while registering "
                    + "ServiceConfigurationLoader.");
        }else{
            theLogger.log(Level.INFO, 
                    "ServiceConfigurationLoader registered successfully.");
        }
        return reg;
    }
    
    public static ServiceRegistration registerConfigurationWriter(
            BundleContext context, ServiceConfigurationWriter writer, 
            Properties props){
        if(context == null || writer == null){
            throw new NullPointerException();
        }
        if(props == null){
            props = new Properties();
        }
        props.put(ServiceConfigurationWriter.PROP_CONFIG_FORMAT_VERSION, 
                writer.getConfigurationFormat().toString());
        props.put(ServiceConfigurationWriter.PROP_SERVICE_CONFIG_CLASS, 
                writer.getConfigurationClass().getName());
        props.put(ServiceConfigurationWriter.PROP_PARAMETER_CLASS, 
                writer.getParameterClass().getName());
        ServiceRegistration reg = context.registerService(
                ServiceConfigurationWriter.class.getName(), writer, props);
        if(reg == null){
            theLogger.log(Level.WARNING, 
                    "Unknown error occured while registering "
                    + "ServiceConfigurationWriter.");
        }else{
            theLogger.log(Level.INFO, 
                    "ServiceConfigurationWriter registered successfully.");
        }
        return reg;
    }
    
    public static <Conf,Param> 
            SingleServiceListener<ServiceConfigurationWriter<Conf,Param>> 
            createWriterServiceListener(BundleContext context, 
                    Class<Conf> configClass,
                    Class<Param> paramClass,
                    VersionProperty configFormat,
                    String serviceFilter){
        Properties props = new Properties();
        props.put(ServiceConfigurationWriter.PROP_CONFIG_FORMAT_VERSION, 
                configFormat.toString());
        props.put(ServiceConfigurationWriter.PROP_SERVICE_CONFIG_CLASS, 
                configClass.getName());
        props.put(ServiceConfigurationWriter.PROP_PARAMETER_CLASS, 
                paramClass.getName());
        String writerFilter = OSGiUtils.createServiceFilter(props);
        if(serviceFilter != null && !serviceFilter.isEmpty()){
            writerFilter = "(&"+writerFilter+"("+serviceFilter+"))";
        }
        return new SingleServiceListener(
                ServiceConfigurationWriter.class, context, serviceFilter);
    }
}
