/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.triggering;

import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.robokind.api.common.utils.TimeUtils;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DefaultTriggerSequence implements TriggerSequence{
    private final static Logger theLogger = 
            Logger.getLogger(DefaultTriggerSequence.class.getName());
    private BundleContext myContext;
    private SortedSet<SequencedTrigger> myTriggers;
    private Properties myProperties;
    private boolean myExecutingFlag;
    private ScheduledTriggerExecutor myExecutor;
    private Thread.UncaughtExceptionHandler myExceptionHandler;
    
    public DefaultTriggerSequence(BundleContext context){
        if(context == null){
            throw new NullPointerException();
        }
        myTriggers = new TreeSet<SequencedTrigger>();
        myExecutingFlag = false;
        myContext = context;
    }
    
    public void setExceptionHandler(Thread.UncaughtExceptionHandler handler){
        myExceptionHandler = handler;
    }
    
    public void addTrigger(long triggerDelayMillisec, Trigger trigger){
        if(myExecutingFlag){
            throw new IllegalStateException("Cannot add Trigger while running.");
        }
        if(trigger == null){
            throw new NullPointerException();
        }
        myTriggers.add(new SequencedTrigger(triggerDelayMillisec, trigger));
    }
    
    @Override
    public SortedSet<SequencedTrigger> getTriggers() {
        return myTriggers;
    }

    @Override
    public void start() {
        if(myExecutingFlag){
            return;
        }
        myExecutingFlag = true;
        myExecutor = new ScheduledTriggerExecutor();
        myExecutor.start();
    }

    @Override
    public void stop() {
        myExecutor.stop();
        myExecutingFlag = false;
    }
    
    class ScheduledTriggerExecutor {
        private ScheduledExecutorService myExecutor;
        
        void start(){
            myExecutor = new ScheduledThreadPoolExecutor(1);
            for(SequencedTrigger trigger : myTriggers){
                TriggerTask task = 
                        new TriggerTask(trigger.getTrigger(), myProperties);
                myExecutor.schedule(task, trigger.getTriggerTimeMillisec(), 
                        TimeUnit.MILLISECONDS);
            }
        }
        
        void stop(){
            try{
                myExecutor.shutdownNow();
                myExecutor = null;
            }catch(SecurityException ex){
                theLogger.log(Level.WARNING, 
                        "Exception caught while stopping TriggerSequence.", ex);
            }
        }
    }
    
    class TriggerTask implements Callable<ServiceRegistration> {
        private Trigger myTrigger;
        private Properties myRegistrationProperties;

        public TriggerTask(Trigger trigger, Properties registrationProperties) {
            myTrigger = trigger;
            myRegistrationProperties = registrationProperties;
        }

        @Override
        public ServiceRegistration call() {
            try{
                theLogger.log(Level.INFO, 
                        "Registering trigger.  type: {0}, trigger: {1}.", 
                        new Object[]{myTrigger.getTriggerType(), myTrigger});
                myTrigger.setTimestampMillisecUTC(TimeUtils.now());
                return myContext.registerService(
                                Trigger.class.getName(), 
                                myTrigger, myRegistrationProperties);
            }catch(Exception ex){
                theLogger.log(Level.WARNING, 
                        "Caught exception in executor.", ex);
                return null;
            }
        }
    }
}
