/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.triggering;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.robokind.api.common.osgi.OSGiUtils;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class OSGiTriggeringService {
    private final static Logger theLogger = Logger.getLogger(OSGiTriggeringService.class.getName());
    private BundleContext myContext;
    private OSGiTriggerListener myListener;
    
    public OSGiTriggeringService(BundleContext context){
        if(context == null){
            throw new NullPointerException();
        }
        myContext = context;
        myListener = new OSGiTriggerListener();
        try{
            OSGiUtils.addServiceListener(
                    myContext, myListener, Trigger.class, null);
        }catch(InvalidSyntaxException ex){
            //null can't be invalid
        }
    }
    
    public void triggerEvent(Trigger trigger){
        theLogger.log(Level.WARNING, 
                "Triggering event.  type: {0}, filter: {1}.", 
                new Object[]{
                    trigger.getTriggerType(), trigger.getTypeFilter()});
        ServiceReference[] refs = TriggerUtils.findHandlers(myContext, trigger);
        if(refs != null){
            for(ServiceReference ref : refs){
                if(triggerEvent(ref, trigger)){
                    return;
                }
            }
        }
        theLogger.log(Level.WARNING, 
                "Unable to trigger event, no handler available.  "
                + "type: {0}, filter: {1}.", 
                new Object[]{
                    trigger.getTriggerType(), trigger.getTypeFilter()});
    }
    
    private boolean triggerEvent(ServiceReference ref, Trigger trigger){
        TriggerHandler handler = 
                OSGiUtils.getService(TriggerHandler.class, myContext, ref);
        if(handler == null){
            return false;
        }
        handler.handleTrigger(trigger);
        myContext.ungetService(ref);
        return true;
    }
    
    
    
    class OSGiTriggerListener implements ServiceListener {
        @Override
        public void serviceChanged(ServiceEvent event) {
            if(ServiceEvent.REGISTERED != event.getType()){
                return;
            }
            theLogger.info("Received Trigger ServiceEvent.");
            ServiceReference ref = event.getServiceReference();
            Trigger trigger = 
                    OSGiUtils.getService(Trigger.class, myContext, ref);
            if(trigger == null){
                theLogger.warning(
                        "Could not retrieve Trigger from ServiceReference.");
                return;
            }
            triggerEvent(trigger);
        }
    }
}
