/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.common.triggering;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.robokind.api.common.osgi.OSGiUtils;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class TriggerUtils {
    private final static Logger theLogger = 
            Logger.getLogger(TriggerUtils.class.getName());
    
    public static ServiceRegistration registerTriggerHandler(BundleContext context, TriggerHandler handler, Properties props){
        if(context == null || handler == null){
            throw new NullPointerException();
        }
        return OSGiUtils.registerService(context, 
                TriggerHandler.class.getName(), 
                Trigger.PROP_TRIGGER_TYPE, 
                handler.getTriggerType(), 
                handler, props);
    }
    
    public static ServiceReference[] findHandlers(
            BundleContext context, Trigger trigger){
        return findHandlers(
                context, trigger.getTriggerType(), trigger.getTypeFilter());
    }
    
    public static ServiceReference[] findHandlers(
            BundleContext context, String triggerType, String filter){
        if(context == null || triggerType == null){
            throw new NullPointerException();
        }
        String combinedFilter = OSGiUtils.createIdFilter(
                Trigger.PROP_TRIGGER_TYPE, triggerType, filter);
        ServiceReference[] refs;
        try{
            refs = context.getServiceReferences(
                    TriggerHandler.class.getName(), combinedFilter);
        }catch(InvalidSyntaxException ex){
            theLogger.log(Level.WARNING, 
                    "Error finding TriggerHandler, invalid filter: " + 
                    combinedFilter + ".", ex);
            return null;
        }
        if(refs == null){
            theLogger.log(Level.WARNING, 
                    "Could not find TriggerHandler matching filter: {0}.", 
                    combinedFilter);
        }
        return refs;
    }
}
