/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.messaging;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;

/**
 * Default implementation of the MessageBlockingReceiver.
 * Blocks until a Record is received.  Uses an Adapter to adapt Records to 
 * Messages and notifies Listeners.
 * 
 * @param <Msg> type of message to return
 * @param <Rec> type of record to receive
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DefaultMessageBlockingReceiver<Msg, Rec> 
        implements MessageBlockingReceiver<Msg> {
    private final static Logger theLogger = 
            Logger.getLogger(DefaultMessageBlockingReceiver.class.getName());
    private RecordBlockingReceiver<Rec> myRecordReceiver;
    private Adapter<Rec,Msg> myAdapter;
    private List<Listener<Msg>> myListeners;
    
    /**
     * Creates an empty DefaultMessageBlockingReceiver.
     */
    public DefaultMessageBlockingReceiver(){
        myListeners = new ArrayList<Listener<Msg>>();
    }
    
    /**
     * Sets the RecordAsyncReceiver used to receive Records.
     * @param service theRecordReceiver to set
     */
    public void setRecordReceiver(RecordBlockingReceiver<Rec> receiver){
        myRecordReceiver = receiver;
    }
    /**
     * Sets the Adapter used to convert Records to Messages.
     * @param adapter  the Adapter to set
     */
    public void setAdapter(Adapter<Rec,Msg> adapter){
        myAdapter = adapter;
    }

    @Override
    public void start(){
        if(myRecordReceiver == null){
            theLogger.warning("No Record PollingService, unable to receive.");
            return;
        }else if(myAdapter == null){
            theLogger.warning("No Record Adapter, unable to send receive.");
            return;
        }
    }

    @Override
    public void stop() {}

    @Override
    public Msg fetchMessage(long timeout){
        if(myRecordReceiver == null || myAdapter == null){
            throw new NullPointerException();
        }
        Rec rec = myRecordReceiver.fetchRecord(timeout);
        if(rec == null){
            return null;
        }
        Msg msg = myAdapter.adapt(rec);
        if(msg != null){
            fireMessageEvent(msg);
        }
        return msg;
    }
    /**
     * Notifies listeners of a Message
     * @param message 
     */
    protected void fireMessageEvent(Msg message){
        for(Listener<Msg> listener : myListeners){
            listener.handleEvent(message);
        }
    }

    @Override
    public int clearMessages() {
        return myRecordReceiver.clearRecords();
    }

    @Override
    public void addMessageListener(Listener<Msg> listener) {
        if(listener == null){
            return;
        }
        if(!myListeners.contains(listener)){
            myListeners.add(listener);
        }
    }

    @Override
    public void removeMessageListener(Listener<Msg> listener) {
        if(listener == null){
            return;
        }
        myListeners.remove(listener);
    }
}
