/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.api.motion.blending;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import org.robokind.api.common.position.NormalizedDouble;
import org.robokind.api.common.utils.Factory;
import org.robokind.api.motion.protocol.FrameSource;
import org.robokind.api.motion.protocol.MotionFrame;
import org.robokind.api.motion.protocol.JointPositionMap;
import org.robokind.api.common.utils.Utils;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class NaiveMotionFrameAverager<
        Id, PosMap extends JointPositionMap<Id,NormalizedDouble>> implements 
        FrameCombiner<MotionFrame<PosMap>,FrameSource<PosMap>,PosMap> {
    private Factory<PosMap> myPositionMapFactory;    
    
    public NaiveMotionFrameAverager(Factory<PosMap> posMapFact){
        if(posMapFact == null){
            throw new NullPointerException();
        }
        myPositionMapFactory = posMapFact;
    }
    /**
     * Returns a JointPositionMap, averaging the goal positions from the given 
     * MotionFrames.
     * @return JointPositionMap, averaging the goal positions from the given 
     * MotionFrames
     */
    @Override
    public PosMap combineFrames(long time, long interval, PosMap curPos, 
    Map<? extends MotionFrame<PosMap>, ? extends FrameSource<PosMap>> frames) {
        //Use a HashMap instead of a PositionMap since the sum may be greater
        //than 1.0 before averaging.
        Map<Id, Double> posSums = new HashMap();
        Map<Id, Integer> count = new HashMap();
        
        for(MotionFrame f : frames.keySet()){
            sumGoalPositions(f, posSums, count);
        }
        PosMap pos = myPositionMapFactory.create();
        for(Entry<Id,Double> e : posSums.entrySet()){
            Id id = e.getKey();
            double goal = e.getValue();
            double c = count.get(id);
            double avg = goal/c;
            avg = Utils.bound(avg, 0.0, 1.0);
            NormalizedDouble val = new NormalizedDouble(avg);
            pos.put(id, val);
        }
        return pos;
    }

    private void sumGoalPositions(MotionFrame<PosMap> f, 
            Map<Id, Double> posSums, Map<Id, Integer> count) {
        PosMap goals = f.getGoalPositions();
        if(goals == null){
            return;
        }
        for(Entry<Id,NormalizedDouble> e : goals.entrySet()){
            Id id = (Id)e.getKey();
            NormalizedDouble goal = e.getValue();
            double val = goal.getValue();
            int c = 1;
            if(posSums.containsKey(id)){
                val += posSums.get(id);
                c += count.get(id);
            }
            posSums.put(id, val);
            count.put(id, c);
        }
    }
}
