/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.api.motion.jointgroup;

import java.util.List;
import org.osgi.framework.BundleContext;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Joint;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.api.common.osgi.OSGiUtils;
import org.robokind.api.common.osgi.ServiceClassListener;
import org.robokind.api.motion.Robot.JointId;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RobotJointGroup extends 
        AbstractJointGroup<JointId, RobotJointGroup, Joint> {
    /**
     * Service version name.
     */
    public final static String CONFIG_TYPE = "RobotJointGroup";
    /**
     * Service version number.
     */
    public final static String CONFIG_VERSION = "1.0";
    /**
     * Service VersionProperty.
     */
    public final static VersionProperty VERSION = 
            new VersionProperty(CONFIG_TYPE, CONFIG_VERSION);
    
    private Robot myRobot;
    
    private ServiceClassListener<Robot> myRobotListener;

    public RobotJointGroup(String name, 
            List<JointId> ids, List<RobotJointGroup> groups){
        super(name, ids, groups);
    }
    
    private void setRobot(Robot robot){
        Robot old = myRobot;
        setRobotQuite(robot);
        firePropertyChange(PROP_STRUCTURE_CHANGED, old, myRobot);
    }
    
    private void setRobotQuite(Robot robot){
        if(myRobot == robot){
            return;
        }
        myRobot = robot;
        for(RobotJointGroup group : myGroups){
            group.setRobotQuite(robot);
        }
    }
    
    protected void initRobotListener(Robot.Id robotId){
        BundleContext context = OSGiUtils.getBundleContext(Robot.class);
        String filter = OSGiUtils.createServiceFilter(
                Robot.PROP_ID, robotId.toString());
        myRobotListener = new ServiceClassListener<Robot>(
                Robot.class, context, filter) {

            @Override
            protected void addService(Robot t) {
                setGroupRobot();
            }

            @Override
            protected void removeService(Robot t) {
                setGroupRobot();
            }
            
            private void setGroupRobot(){
                if(myList.isEmpty()){
                    setRobot(null);
                    return;
                }
                Robot robot = myList.get(0);
                setRobot(robot);
            }
        };
    }
    
    @Override
    protected Joint getJointById(JointId jointId) {
        if(myRobot == null){
            return null;
        }
        return myRobot.getJoint(jointId);
    }
}
