/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.motion.messaging;

import java.util.logging.Level;
import org.robokind.api.motion.messaging.messages.RobotRequestFactory;
import org.robokind.api.motion.messaging.messages.RobotDefinitionResponse;
import org.robokind.api.motion.messaging.messages.RobotRequest;
import java.util.logging.Logger;
import org.robokind.api.common.utils.Listener;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.messaging.MessageBlockingReceiver;
import org.robokind.api.messaging.MessageSender;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.JointId;
import org.robokind.api.motion.Robot.RobotPositionMap;
import org.robokind.api.motion.messaging.messages.RobotResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotPositionResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotResponseHeader;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotStatusResponse;
import org.robokind.api.motion.protocol.MotionFrame;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RemoteRobotClient {
    private final static Logger theLogger = 
            Logger.getLogger(RemoteRobotClient.class.getName());
    
    public final static int DEFAULT_TIMEOUT_LENGTH = 10000;
    
    private Robot.Id myRobotId;
    private String mySourceId;
    private String myDestinationId;
    private RobotRequestFactory myRequestFactory;
    private MessageSender<RobotRequest, ?, Listener<RobotRequest>> myRequestSender;
    private MessageBlockingReceiver<RobotResponse, ?, Listener<RobotResponse>> myResponseReceiver;
    private MotionFrameSender myMotionFrameSender;
    
    public RemoteRobotClient(Robot.Id robotId, 
            String sourceId, String destId, RobotRequestFactory reqFact){
        if(robotId == null || sourceId == null || 
                destId == null || reqFact == null){
            throw new NullPointerException();
        }
        myRobotId = robotId;
        mySourceId = sourceId;
        myDestinationId = destId;
        myRequestFactory = reqFact;
    }
    
    public void setRequestSender(MessageSender<
            RobotRequest, ?, Listener<RobotRequest>> reqSender){
        if(reqSender == null){
            throw new NullPointerException();
        }
        myRequestSender = reqSender;
    }
    
    public void setResponseReceiver(MessageBlockingReceiver<
            RobotResponse, ?, Listener<RobotResponse>> respRec){
        if(respRec == null){
            throw new NullPointerException();
        }
        myResponseReceiver = respRec;
    }
    
    public void setMotionFrameSender(MotionFrameSender frameSender){
        if(frameSender == null){
            throw new NullPointerException();
        }
        myMotionFrameSender = frameSender;
    }
    
    public Robot.Id getRobotId(){
        return myRobotId;
    }
    
    public String getSourceId(){
        return mySourceId;
    }
    
    public String getDestinationId(){
        return myDestinationId;
    }
    
    public RobotDefinitionResponse requestRobotDefinition(){
        return makeDefinitionRequest(
                RobotRequest.CMD_GET_ROBOT_DEFINITION, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendConnect(){
        return makeStatusRequest(
                RobotRequest.CMD_CONNECT_ROBOT, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendDisconnect(){
        return makeStatusRequest(
                RobotRequest.CMD_DISCONNECT_ROBOT, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean getConnected(){
        return makeStatusRequest(
                RobotRequest.CMD_GET_CONNECTION_STATUS, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendEnable(){
        return makeStatusRequest(
                RobotRequest.CMD_ENABLE_ROBOT, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendDisable(){
        return makeStatusRequest(
                RobotRequest.CMD_DISABLE_ROBOT, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean getEnabled(){
        return makeStatusRequest(
                RobotRequest.CMD_GET_ENABLED_STATUS, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendJointEnable(JointId jointId){
        return makeStatusRequestForJoint(
                RobotRequest.CMD_ENABLE_JOINT, jointId, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean sendJointDisable(JointId jointId){
        return makeStatusRequestForJoint(
                RobotRequest.CMD_DISABLE_JOINT, 
                jointId, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public boolean getJointEnabled(JointId jointId){
        return makeStatusRequestForJoint(
                RobotRequest.CMD_GET_JOINT_ENABLED_STATUS, 
                jointId, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public RobotPositionMap requestDefaultPositions(){
        return makePositionRequest(
                RobotRequest.CMD_GET_DEFAULT_POSITIONS, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public RobotPositionMap requestGoalPositions(){
        return makePositionRequest(
                RobotRequest.CMD_GET_GOAL_POSITIONS, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public RobotPositionMap requestCurrentPositions(){
        return makePositionRequest(
                RobotRequest.CMD_GET_CURRENT_POSITIONS, DEFAULT_TIMEOUT_LENGTH);
    }
    
    public void sendMovement(MotionFrame frame){
        myMotionFrameSender.sendMotionFrame(frame);
    }    
    
    private Boolean makeStatusRequest(String requestType, long timeout){
        RobotStatusResponse resp = makeBlockingRequest(
            RobotStatusResponse.class, requestType, timeout);
        if(resp == null){
            theLogger.log(Level.WARNING, "Received null status for: {0}, from: {1}", 
                    new Object[]{requestType, myRobotId.getRobtIdString()});
            return null;
        }
        return resp.getStatusResponse();
    }
    
    private Boolean makeStatusRequestForJoint(String requestType, JointId jointId, long timeout){
        RobotStatusResponse resp = makeBlockingRequestForJoint(
            requestType, jointId, timeout);
        if(resp == null){
            theLogger.log(Level.WARNING, "Received null positions for: {0}, from: {1}", 
                    new Object[]{requestType, myRobotId.getRobtIdString()});
            return null;
        }
        return resp.getStatusResponse();
    }
    
    private RobotPositionMap makePositionRequest(
            String requestType, long timeout){
        RobotPositionResponse resp = makeBlockingRequest(
                RobotPositionResponse.class, requestType, timeout);
        if(resp == null){
            theLogger.log(Level.WARNING, "Received null positions for: {0}, from: {1}", 
                    new Object[]{requestType, myRobotId.getRobtIdString()});
            return null;
        }
        return resp.getPositionMap();
    }
    
    private RobotDefinitionResponse makeDefinitionRequest(
            String requestType, long timeout){
        RobotDefinitionResponse resp = makeBlockingRequest(
                RobotDefinitionResponse.class, requestType, timeout);
        if(resp == null){
            theLogger.log(Level.WARNING, "Received null definition for: {0}, from: {1}", 
                    new Object[]{requestType, myRobotId.getRobtIdString()});
            return null;
        }
        return resp;
    }
    
    private synchronized <T extends RobotResponse> T makeBlockingRequest(
            Class<T> responseClass, String requestType, long timeout){
        if(responseClass == null || requestType == null){
            throw new NullPointerException();
        }
        myResponseReceiver.clearMessages();
        RobotRequest req = myRequestFactory.buildRobotRequest(
                myRobotId, mySourceId, myDestinationId, 
                requestType, TimeUtils.now());
        myRequestSender.sendMessage(req);
        return fetchTypedResponse(responseClass, req, timeout);
    }
    
    private synchronized RobotStatusResponse makeBlockingRequestForJoint(
            String requestType, JointId jointId, long timeout){
        if(requestType == null){
            throw new NullPointerException();
        }
        RobotRequest req = myRequestFactory.buildJointRequest(
                jointId, mySourceId, myDestinationId, 
                requestType, TimeUtils.now());
        myRequestSender.sendMessage(req);
        return fetchTypedResponse(RobotStatusResponse.class, req, timeout);
    }
    
    private <T extends RobotResponse> T fetchTypedResponse(
            Class<T> clazz, RobotRequest request, long timeout){
        long start = TimeUtils.now();
        do{
            RobotResponse resp = 
                    (RobotResponse)myResponseReceiver.fetchMessage(timeout);
            if(resp == null){
                theLogger.warning("Received null Message from Receiver");
            }else if(!isMatch(request, resp.getResponseHeader())){
                theLogger.warning("Response does not match Request.  Ignoring response.");
            }else if(clazz.isAssignableFrom(resp.getClass())){
                return (T)resp;
            }
            long elapsed = TimeUtils.now() - start;
            timeout -= elapsed;
        }while(timeout > 0);
        return null;
    }
    
    private boolean isMatch(RobotRequest req, RobotResponseHeader resp){
        if(req == null || resp == null){
            throw new NullPointerException();
        }
        if(!req.getRobotId().equals(resp.getRobotId())){
            return false;
        }else if(!req.getRequestType().equals(resp.getRequestType())){
            return false;
        /*}else if(!req.getSourceId().equals(resp.getDestinationId())){
            return false;
        }else if(!req.getDestinationId().equals(resp.getSourceId())){
            return false;
        */}else if(req.getTimestampMillisecUTC() != resp.getRequestTimestampMillisecUTC()){
            return false;
        }
        return true;
    }
}
