/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.motion.messaging;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.robokind.api.motion.messaging.messages.RobotRequest;
import org.robokind.api.motion.messaging.messages.RobotResponseFactory;
import org.robokind.api.common.utils.Listener;
import org.robokind.api.messaging.MessageAsyncReceiver;
import org.robokind.api.messaging.MessageSender;
import org.robokind.api.motion.Joint;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.JointId;
import org.robokind.api.motion.Robot.RobotPositionMap;
import org.robokind.api.motion.messaging.messages.RobotResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotResponseHeader;
import org.robokind.api.motion.protocol.MotionFrameEvent;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RemoteRobotHost {
    private final static Logger theLogger = 
            Logger.getLogger(RemoteRobotHost.class.getName());
    private Robot myRobot;
    private String mySourceId;
    private String myDestinationId;
    private MessageSender<RobotResponse, ?, Listener<RobotResponse>> myResponseSender;
    private MessageAsyncReceiver<RobotRequest, ?, Listener<RobotRequest>> myRequestReceiver;
    private RequestListener myRequestListener;
    private RobotResponseFactory myResponseFactory;
    private MessageAsyncReceiver<MotionFrameEvent, ?, Listener<MotionFrameEvent>> myMotionFrameReceiver;
    private Listener<MotionFrameEvent> myMoveHandler;
    private boolean myStartFlag;
    
    public RemoteRobotHost(Robot robot, 
            String sourceId, String destinationId, 
            MessageSender<RobotResponse, ?, Listener<RobotResponse>> sender, 
            MessageAsyncReceiver<RobotRequest, ?, Listener<RobotRequest>> receiver, 
            RobotResponseFactory factory, 
            MessageAsyncReceiver<MotionFrameEvent, ?, Listener<MotionFrameEvent>> motionFrameReceiver, 
            Listener<MotionFrameEvent> moveHandler){
        this(sourceId, destinationId);
        initialize(robot, sender, receiver, factory, motionFrameReceiver, moveHandler);
    }
    
    protected RemoteRobotHost(String sourceId, String destinationId){
        if(sourceId == null || destinationId == null){
            throw new NullPointerException();
        }
        myStartFlag = false;
        mySourceId = sourceId;
        myDestinationId = destinationId;
    }
    
    private void initialize(
            Robot robot, 
            MessageSender<RobotResponse, ?, Listener<RobotResponse>> sender, 
            MessageAsyncReceiver<RobotRequest, ?, Listener<RobotRequest>> receiver, 
            RobotResponseFactory factory, 
            MessageAsyncReceiver<MotionFrameEvent, ?, Listener<MotionFrameEvent>> motionFrameReceiver, 
            Listener<MotionFrameEvent> moveHandler){
        if(robot == null || 
                sender == null || receiver == null || factory == null){
            throw new NullPointerException();
        }
        myRobot = robot;
        myResponseSender = sender;
        myRequestReceiver = receiver;
        myResponseFactory = factory;
        myMotionFrameReceiver = motionFrameReceiver;
        myMoveHandler = moveHandler;
        
    }
    
    protected void setRobot(Robot robot){
        if(robot == null){
            throw new NullPointerException();
        }
        myRobot = robot;
    }
    
    protected void setResponseSender(MessageSender<
            RobotResponse, ?, Listener<RobotResponse>> sender){
        if(sender == null){
            throw new NullPointerException();
        }
        myResponseSender = sender;
    }
    
    protected void setRequestReceiver(MessageAsyncReceiver<
            RobotRequest, ?, Listener<RobotRequest>> receiver){
        if(receiver == null){
            throw new NullPointerException();
        }
        myRequestReceiver = receiver;
    }
    
    protected void setResponseFactory(RobotResponseFactory factory){
        if(factory == null){
            throw new NullPointerException();
        }
        myResponseFactory = factory;
    }
    
    protected void setMotionFrameReceiver(MessageAsyncReceiver<
            MotionFrameEvent, ?, Listener<MotionFrameEvent>> receiver){
        if(receiver == null){
            throw new NullPointerException();
        }
        myMotionFrameReceiver = receiver;
    }
    
    protected void setMoveHandle(Listener<MotionFrameEvent> moveHandler){
        if(moveHandler == null){
            throw new NullPointerException();
        }
        myMoveHandler = moveHandler;
    }
    
    public Robot.Id getRobotId(){
        return myRobot.getRobotId();
    }
    
    public String getSourceId(){
        return mySourceId;
    }
    
    public String getDestinationId(){
        return myDestinationId;
    }
    
    public void start(){
        if(myRequestReceiver == null || myMotionFrameReceiver == null || myMoveHandler == null){
            throw new NullPointerException();
        }else if(myStartFlag){
            return;
        }
        if(myRequestListener == null){
            myRequestListener = new RequestListener();
            myRequestReceiver.addMessageListener(myRequestListener);
        }
        myMotionFrameReceiver.addMessageListener(myMoveHandler);
        myStartFlag = true;
    }
    
    private Robot getRobot(){
        return myRobot;
    }
    
    protected void handleDefinitionRequest(RobotRequest req){
        sendDefinitionResponse(req);
    }
    protected void handleConnectRequest(RobotRequest req){
        sendStatusResponse(req, getRobot().connect());
    }
    protected void handleDisconnectRequest(RobotRequest req){
        getRobot().disconnect();
        sendStatusResponse(req, true);
    }
    protected void handleConnectionStatusRequest(RobotRequest req){
        sendStatusResponse(req, getRobot().isConnected());
    }
    protected void handleEnableRequest(RobotRequest req){
        getRobot().setEnabled(true);
        sendStatusResponse(req, true);
    }
    protected void handleDisableRequest(RobotRequest req){
        getRobot().setEnabled(false);
        sendStatusResponse(req, true);
    }
    protected void handleEnabledStatusRequest(RobotRequest req){
        sendStatusResponse(req, getRobot().isEnabled());
    }
    protected void handleEnableRequestForJoint(RobotRequest req){
        getRequestedJoint(req).setEnabled(true);
        sendStatusResponse(req, true);
    }
    protected void handleDisableRequestForJoint(RobotRequest req){
        getRequestedJoint(req).setEnabled(false);
        sendStatusResponse(req, true);
    }
    protected void handleEnabledStatusRequestForJoint(RobotRequest req){
        sendStatusResponse(req, getRequestedJoint(req).getEnabled());
    }
    protected Joint getRequestedJoint(RobotRequest req){
        Integer jIdInt = req.getRequestIndex();
        if(jIdInt == null){
            throw new NullPointerException();
        }
        Joint.Id jId = new Joint.Id(jIdInt);
        Robot.Id rId = req.getRobotId();
        JointId jointId = new Robot.JointId(rId, jId);
        Joint j = getRobot().getJoint(jointId);
        if(j == null){
            throw new NullPointerException();
        }
        return j;
    }
    
    protected void handleDefaultPositionRequest(RobotRequest req){
        sendPositionResponse(req, getRobot().getDefaultPositions());
    }
    protected void handleGoalPositionRequest(RobotRequest req){
        sendPositionResponse(req, getRobot().getGoalPositions());
    }
    protected void handleCurrentPositionRequest(RobotRequest req){
        sendPositionResponse(req, getRobot().getCurrentPositions());
    }
    private void sendDefinitionResponse(RobotRequest req){
        myResponseSender.sendMessage(
                myResponseFactory.createDefinitionResponse(
                    getHeader(req), 
                    getRobot()));
    }
    private void sendStatusResponse(RobotRequest req, boolean value){
        myResponseSender.sendMessage(
                myResponseFactory.createStatusResponse(
                    getHeader(req), 
                    value));
    }
    private void sendPositionResponse(
            RobotRequest req, RobotPositionMap positions){
        myResponseSender.sendMessage(
                myResponseFactory.createPositionResponse(
                    getHeader(req), 
                    positions));
    }
    private RobotResponseHeader getHeader(RobotRequest req){
        return myResponseFactory.createHeader(
                getRobotId(), mySourceId, myDestinationId,
                req.getRequestType(), req.getTimestampMillisecUTC());
    }
    
    class RequestListener implements Listener<RobotRequest>{

        @Override
        public void handleEvent(RobotRequest event) {
            String reqType = event.getRequestType();
            if(reqType.equals(RobotRequest.CMD_GET_ROBOT_DEFINITION)){
                handleDefinitionRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_CONNECT_ROBOT)){
                handleConnectRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_DISCONNECT_ROBOT)){
                handleDisconnectRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_CONNECTION_STATUS)){
                handleConnectionStatusRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_ENABLE_ROBOT)){
                handleEnableRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_DISABLE_ROBOT)){
                handleDisableRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_ENABLED_STATUS)){
                handleEnabledStatusRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_DEFAULT_POSITIONS)){
                handleDefaultPositionRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_GOAL_POSITIONS)){
                handleGoalPositionRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_CURRENT_POSITIONS)){
                handleGoalPositionRequest(event);
            }else if(reqType.equals(RobotRequest.CMD_ENABLE_JOINT)){
                handleEnableRequestForJoint(event);
            }else if(reqType.equals(RobotRequest.CMD_DISABLE_JOINT)){
                handleDisableRequestForJoint(event);
            }else if(reqType.equals(RobotRequest.CMD_GET_JOINT_ENABLED_STATUS)){
                handleEnabledStatusRequestForJoint(event);
            }else{
                theLogger.log(Level.WARNING, 
                        "Received unknown request type: {0}", reqType);
            }
        }
    }
}
