/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.motion.sync;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.api.common.position.NormalizedDouble;
import org.robokind.api.motion.AbstractRobot;
import org.robokind.api.motion.Joint;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.RobotPositionMap;

/**
 * Holds multiple Robots and synchronizes their movements for the given 
 * JointIds.
 * Events and JointProperties from the primary Robot are forwarded.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class SynchronizedRobot extends AbstractRobot<SynchronizedJoint> {
    /**
     * Robot type version name.
     */
    public final static String VERSION_NAME = "SynchronizedRobot";
    /**
     * Robot type version number.
     */
    public final static String VERSION_NUMBER = "1.0";
    /**
     * Robot type VersionProperty.
     */
    public final static VersionProperty VERSION = new VersionProperty(VERSION_NAME, VERSION_NUMBER);
    
    public final static String PROP_ADD_ROBOT = "addRobot";
    public final static String PROP_REMOVE_ROBOT = "removeRobot";
    
    
    
    private List<Robot> myRobots;
    private Robot myPrimaryRobot;
    
    public SynchronizedRobot(SynchronizedRobotConfig config){
        super(config.getRobotId());
        myRobots = new ArrayList<Robot>();
        for(SynchronizedJointConfig conf : config.getJointConfigs()){
            addJoint(new SynchronizedJoint(conf, this));
        }
    }
    
    public SynchronizedRobot(Robot.Id robotId, Set<Joint.Id> jointIds){
        super(robotId);
        myRobots = new ArrayList<Robot>();
        for(Joint.Id jId : jointIds){
            addJoint(new SynchronizedJoint(jId, this));
        }
    }
    
    @Override
    public boolean connect() {
        return true;
    }

    @Override
    public void disconnect() {}

    @Override
    public boolean isConnected() {
        return true;
    }

    @Override
    public void move(RobotPositionMap positions, long lenMillisec) {
        for(Robot r : myRobots){
            RobotPositionMap newPos = changeId(positions, r.getRobotId());
            r.move(newPos, lenMillisec);
        }
        for(Entry<Robot.JointId,NormalizedDouble> e : positions.entrySet()){
            Robot.JointId jId = e.getKey();
            NormalizedDouble d = e.getValue();
            SynchronizedJoint j = getJoint(jId);
            j.setGoalPosition(d);
        }
    }
    
    private RobotPositionMap changeId(RobotPositionMap pos, Robot.Id newId){
        RobotPositionMap newPos = new RobotPositionHashMap(pos.size());
        for(Entry<Robot.JointId,NormalizedDouble> e : pos.entrySet()){
            Robot.JointId oldJId = e.getKey();
            if(!myJointMap.containsKey(oldJId)){
                continue;
            }
            Robot.JointId jId = new JointId(newId, oldJId.getJointId());
            newPos.put(jId, e.getValue());
        }
        return newPos;
    }
    
    public Robot getPrimaryRobot(){
        return myPrimaryRobot;
    }
    
    public List<Robot> getRobots(){
        return myRobots;
    }
    
    public void addRobot(Robot robot){
        if(robot == null){
            throw new NullPointerException();
        }
        if(robot instanceof SynchronizedRobot){
            throw new IllegalArgumentException("Cannot add SynchronizedRobot.");
        }
        if(!myRobots.contains(robot)){
           return;
        }
        myRobots.add(robot);
        updateJoints();
        firePropertyChange(PROP_ADD_ROBOT, null, robot);
    }
    
    public void removeRobot(Robot.Id robotId){
        if(robotId == null){
            throw new NullPointerException();
        }
        Robot remove = null;
        for(Robot r : myRobots){
            if(robotId.equals(r.getRobotId())){
                remove = r;
            }
        }
        if(remove == null){
            return;
        }
        if(remove.equals(myPrimaryRobot)){
            myPrimaryRobot = null;
        }
        myRobots.remove(remove);
        updateJoints();
        firePropertyChange(PROP_REMOVE_ROBOT, null, remove);
    }
    
    public void setPrimaryRobot(Robot.Id robotId){
        if(robotId == null){
            throw new NullPointerException();
        }
        for(Robot r : myRobots){
            if(robotId.equals(r.getRobotId())){
                myPrimaryRobot = r;
                updatePrimaryJoints();
                return;
            }
        }
    }
    
    private void updateJoints(){
        for(SynchronizedJoint j : myJointList){
            j.updateJointList();
        }
    }
    
    private void updatePrimaryJoints(){
        for(SynchronizedJoint j : myJointList){
            j.updatePrimaryJoint();
        }
    }
}
