/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.motion.utils;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.robokind.api.common.osgi.SingleServiceListener;
import org.robokind.api.common.property.PropertyChangeNotifier;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.RobotPositionMap;

/**
 * Controls a Robot for the UI
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RobotController extends PropertyChangeNotifier implements PropertyChangeListener{
    public final static String PROP_ROBOT = "robot";
    public final static String PROP_SELECT_ROBOT = "selectRobot";
    private BundleContext myContext;
    private Robot.Id myRobotId;
    private SingleServiceListener<Robot> myRobotTracker;
    private PositionTargetFrameSource myRobotMover;
    private ServiceRegistration myMoverRegistration;
    private boolean myListenFlag;
    private RobotManager myManager;
    
    public RobotController(BundleContext context, RobotManager manager){
        if(context == null || manager == null){
            throw new NullPointerException();
        }
        myContext = context;
        myListenFlag = false;
        myManager = manager;
    }
    
    public void setRobotId(Robot.Id robotId){
        Robot oldVal = getRobot();
        unsetRobot();
        myRobotId = robotId;
        if(myRobotId == null){
            firePropertyChange(PROP_ROBOT, oldVal, myRobotId);
            return;
        }
        myRobotTracker = new SingleServiceListener<Robot>(
                Robot.class, myContext, RobotUtils.getRobotFilter(robotId));
        myRobotTracker.start();
        Robot r = getRobot();
        if(r != null){
            r.addPropertyChangeListener(this);
            myListenFlag = true;
        }
        setRobotMover();
        firePropertyChange(PROP_ROBOT, oldVal, myRobotId);
    }
    
    private void unsetRobot(){
        Robot r = getRobot();
        unregisterMover();
        if(r != null){
            r.removePropertyChangeListener(this);
        }
        myListenFlag = false;
        if(myRobotTracker != null){
            myRobotTracker.releaseService();
        }
    }
    
    public Robot getRobot(){
        if(myRobotTracker == null){
            return null;
        }
        Robot r = myRobotTracker.getService();
        if(!myListenFlag && r != null){
            r.addPropertyChangeListener(this);
            myListenFlag = true;
        }
        return r;
    }
    
    public boolean connectRobot(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        if(r.isConnected()){
            return true;
        }
        return r.connect();
    }
    
    public boolean disconnectRobot(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        if(!r.isConnected()){
            return true;
        }
        r.disconnect();
        return !r.isConnected();
    }
    
    public boolean enableRobot(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        if(r.isEnabled()){
            return true;
        }
        r.setEnabled(true);
        return true;
    }
    
    public boolean disableRobot(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        if(!r.isEnabled()){
            return true;
        }
        r.setEnabled(false);
        return true;
    }
    
    public boolean selectRobot(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        return false;
    }
    
    public boolean setDefaultPositions(){
        Robot r = getRobot();
        if(r == null){
            return false;
        }
        RobotPositionMap defPos = r.getDefaultPositions();
        myRobotMover.putPositions(defPos);
        return true;
    }
    
    public boolean removeRobot(){
        unsetRobot();
        return true;
    }
    
    private void setRobotMover(){
        unregisterMover();
        if(myContext == null || myRobotId == null){
            return;
        }
        myRobotMover = new PositionTargetFrameSource(
                myContext, myRobotId, 0.001, null);
        myMoverRegistration = 
                RobotUtils.registerFrameSource(
                myContext, myRobotId, myRobotMover);
    }
    
    private void unregisterMover(){
        if(myMoverRegistration != null){
            myMoverRegistration.unregister();
        }
        myMoverRegistration = null;
        myRobotMover = null;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        firePropertyChange(evt);
    }
}
