/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.motion.lifecycle;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import org.osgi.framework.BundleContext;
import org.robokind.api.common.lifecycle.ServiceLifecycleProvider;
import org.robokind.api.common.lifecycle.utils.ManagedServiceGroup;
import org.robokind.api.common.lifecycle.utils.SimpleLifecycle;
import org.robokind.api.common.osgi.OSGiUtils;
import org.robokind.api.common.osgi.lifecycle.OSGiComponentFactory;
import org.jflux.api.core.util.Factory;
import org.robokind.api.common.utils.Utils;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.JointId;
import org.robokind.api.motion.Robot.RobotPositionHashMap;
import org.robokind.api.motion.Robot.RobotPositionMap;
import org.robokind.api.motion.blending.Blender;
import org.robokind.api.motion.blending.FrameCombiner;
import org.robokind.api.motion.blending.FrameSourceTracker;
import org.robokind.api.motion.blending.NaiveMotionFrameAverager;
import org.robokind.api.motion.blending.OSGiFrameSourceTracker;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class DefaultBlenderServiceGroup extends ManagedServiceGroup {
    private final static Logger theLogger = 
            Logger.getLogger(DefaultBlenderServiceGroup.class.getName());
    
    private static String getIdBase(Robot.Id robotId){
        String base = "robot/" +  robotId + "/blender";
        //TODO: sanitize base
        return base;
    }
    
    public DefaultBlenderServiceGroup(
            BundleContext context, Robot.Id robotId, 
            long blenderIntervalMillisec, Properties registrationProperties){
        super(new OSGiComponentFactory(context), 
                getBlenderLifecycles(context, robotId, blenderIntervalMillisec), 
                getIdBase(robotId), 
                registrationProperties);
    }
    
    private static List<ServiceLifecycleProvider> getBlenderLifecycles(
            BundleContext context, Robot.Id robotId, long blenderInterval){
        blenderInterval = validateInterval(blenderInterval);
        List<ServiceLifecycleProvider> services = new ArrayList();
        services.add(new RobotBlenderLifecycle(robotId));
        services.add(new RobotOutputLifecycle(robotId));
        services.add(new TimedBlenderDriverLifecycle(robotId, blenderInterval));
        services.add(buildFrameSourceTrackerLauncher(context, robotId));
        services.add(buildFrameCombinerLauncher(robotId));
        return services;
    }
    
    private static long validateInterval(long interval){
        return Utils.bound(interval, 1, Integer.MAX_VALUE-1);
    }
    
    private static ServiceLifecycleProvider<FrameSourceTracker> 
            buildFrameSourceTrackerLauncher(
                    BundleContext context, Robot.Id robotId){
        Properties props = new Properties();
        props.put(Robot.PROP_ID, robotId.getRobtIdString());
        OSGiFrameSourceTracker tracker = new OSGiFrameSourceTracker();
        tracker.init(context, OSGiUtils.createServiceFilter(props));
        props.put(Blender.PROP_POSITION_MAP_TYPE, 
                RobotPositionMap.class.getName());
        return new SimpleLifecycle<FrameSourceTracker>(
                        tracker, FrameSourceTracker.class, props);
    }
    
    private static ServiceLifecycleProvider<FrameCombiner> 
            buildFrameCombinerLauncher(Robot.Id robotId){
        Properties props = new Properties();
        props.put(Robot.PROP_ID, robotId.getRobtIdString());
        props.put(Blender.PROP_POSITION_MAP_TYPE, 
                RobotPositionMap.class.getName());
        return new SimpleLifecycle<FrameCombiner>(
                        getFrameCombiner(), FrameCombiner.class, props);
    }
    
    private static FrameCombiner getFrameCombiner(){
        Factory<RobotPositionMap> factory = new Factory<RobotPositionMap>() {
            @Override
            public RobotPositionMap create() {
                return new RobotPositionHashMap();
            }
        };
        return new NaiveMotionFrameAverager<JointId, RobotPositionMap>(factory);
    }
}
