/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.speech.utils;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.robokind.api.common.triggering.Trigger;
import org.robokind.api.common.triggering.TriggerHandler;
import org.robokind.api.speech.SpeechService;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class SpeechTriggerHandler implements TriggerHandler{
    private final static Logger theLogger = 
            Logger.getLogger(SpeechTriggerHandler.class.getName());
    
    public final static String SPEECH_TRIGGER_TYPE = "Speech";
    public final static String PROP_SPEECH_TEXT = "speechText";
    
    private SpeechService mySpeechService;
    
    public SpeechTriggerHandler(SpeechService speechService){
        mySpeechService = speechService;
    }
    
    public void setSpeechService(SpeechService speechService){
        mySpeechService = speechService;
    }
    
    @Override
    public void handleTrigger(Trigger trigger) {
        if(mySpeechService == null){
            theLogger.warning("SpeechService is null.  Ignoring Trigger.");
            return;
        }
        if(trigger == null){
            throw new NullPointerException();
        }
        if(!getTriggerType().equals(trigger.getTriggerType())){
            theLogger.log(Level.WARNING, 
                    "Invalid TriggerType.  Expected: {0}, Found: {1}", 
                    new Object[]{
                        getTriggerType(), trigger.getTriggerType()});
            return;
        }
        Properties props = trigger.getTriggerProperties();
        if(props == null){
            throw new NullPointerException();
        }
        String text = props.getProperty(PROP_SPEECH_TEXT);
        if(text == null){
            throw new NullPointerException();
        }
        mySpeechService.speak(text);
    }

    @Override
    public String getTriggerType() {
        return SPEECH_TRIGGER_TYPE;
    }
}
