/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.api.speech.viseme;

import java.util.ArrayList;
import java.util.List;
import org.jflux.api.core.Listener;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.speech.SpeechEvent;

/**
 * Converts a SpeechEvent to a VisemeEvent and notifies listeners.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class VisemeEventNotifier implements Listener<SpeechEvent>{
    private List<Listener<VisemeEvent>> myListeners;
    private long myRemoteStartTime;
    private long myLocalStartTime;
    private long myTimeOffset;
    
    /**
     * Creates an empty VisemeEventNotifier.
     */
    public VisemeEventNotifier(){
        myListeners = new ArrayList<Listener<VisemeEvent>>();
    }
    
    /**
     * Adds a VisemeEvent listener to be notified.
     * @param listener VisemeEvent listener to add
     */
    public void addListener(Listener<VisemeEvent> listener){
        if(!myListeners.contains(listener)){
            myListeners.add(listener);
        }
    }
    
    /**
     * Removes a VisemeEvent listener.
     * @param listener VisemeEvent listener to removes
     */
    public void removeListener(Listener<VisemeEvent> listener){
        if(myListeners.contains(listener)){
            myListeners.remove(listener);
        }
    }
        
    @Override
    public void handleEvent(SpeechEvent event) {
        if(event == null || event.getSpeechEventType() == null 
                || myListeners.isEmpty()){
            return;
        }
        if(SpeechEvent.SPEECH_START.equals(event.getSpeechEventType())){
            myLocalStartTime = TimeUtils.now();
            myRemoteStartTime = event.getTimestampMillisecUTC();
            myTimeOffset = myLocalStartTime - myRemoteStartTime;
        }else if(SpeechEvent.VISEME.equals(event.getSpeechEventType())){
            handleVisemeEvent(new DefaultVisemeEvent(event, myTimeOffset));
        }
    }

    /**
     * Notifies listeners of a VisemeEvent.
     * @param event VisemeEvent to send to listeners
     */
    public void handleVisemeEvent(VisemeEvent event) {
        for(Listener<VisemeEvent> l : myListeners){
            l.handleEvent(event);
        }
    }
    
}
